/**
 * Copyright 2013 Peergreen SAS
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.ow2.shelbie.core.internal.system;

import java.io.InputStream;
import java.io.PrintStream;

import org.apache.felix.ipojo.annotations.Component;
import org.apache.felix.ipojo.annotations.Provides;
import org.osgi.framework.Bundle;
import org.osgi.framework.BundleContext;
import org.osgi.framework.BundleException;
import org.ow2.shelbie.core.system.SystemService;

/**
 * Default {@link SystemService}.
 */
@Component
@Provides
public class DefaultSystemService implements SystemService {

    private final Bundle system;
    private boolean stopping = false;

    /**
     * InputStream for reading.
     */
    private InputStream in;

    /**
     * PrintStream for output.
     */
    private PrintStream out;

    /**
     * PrintStream for errors.
     */
    private PrintStream err;



    public DefaultSystemService(BundleContext bundleContext) {
        this.system = bundleContext.getBundle(0);

        // Default values
        this.in = System.in;
        this.out = System.out;
        this.err = System.err;
    }

    public void shutdown() {
        try {
            // AFAIU, this is not strictly required to stop bundle[0] in a separate Thread,
            // because the spec says that stop() for the system bundle must spawn a Thread
            // to do its work (and returns as quickly as possible)
            system.stop();
            stopping = true;
        } catch (BundleException e) {
            // Should not happen
        }
    }

    public boolean isStopping() {
        return stopping;
    }



    /**
     * Sets the Stream for reading.
     * @param in the given inputstream
     */
    public void setIn(InputStream in) {
        this.in = in;
    }

    /**
     * @return the inputstream {@link System.in}.
     */
    public InputStream getIn() {
        return in;
    }

    /**
     * Sets the Stream for writing {@link System.out}.
     * @param out the given printstream
     */
    public void setOut(PrintStream out) {
        this.out = out;
    }

    /**
     * @return the printstream {@link System.out}.
     */
    public PrintStream getOut() {
        return out;
    }

    /**
     * Sets the Stream for writing {@link System.err}.
     * @param err the given printstream
     */
    public void setErr(PrintStream err) {
        this.err = err;
    }

    /**
     * @return the printstream {@link System.out}.
     */
    public PrintStream getErr() {
        return err;
    }
}
