/**
 * Copyright 2012 OW2 Shelbie
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.ow2.shelbie.commands.ssh.server.internal;

import org.apache.sshd.common.KeyPairProvider;
import org.apache.sshd.common.keyprovider.FileKeyPairProvider;
import org.apache.sshd.common.util.SecurityUtils;
import org.apache.sshd.server.keyprovider.SimpleGeneratorHostKeyProvider;

import java.io.File;
import java.util.ArrayList;
import java.util.List;

/**
 * @author Loris Bouzonnet
 */
public class KeyPairProviderFactory {

    private final HostKey hostKey;

    private final List<String> stdKeys;
    
    private final String password;

    public KeyPairProviderFactory(HostKey hostKey, final String password) {
        this.hostKey = hostKey;

        stdKeys = new ArrayList<String>(2);

        if (hostKey.getRsaKey() != null && hostKey.getRsaKey().exists()) {
            stdKeys.add(hostKey.getRsaKey().getAbsolutePath());
        }
        if (hostKey.getDsaKey() != null && hostKey.getDsaKey().exists()) {
            stdKeys.add(hostKey.getDsaKey().getAbsolutePath());
        }
        this.password = password;
    }

    public KeyPairProvider get() throws Exception {

        File objKey = hostKey.getObjKey();

        if (objKey != null && objKey.exists()) {
            if (stdKeys.isEmpty()) {
                return new SimpleGeneratorHostKeyProvider(objKey.getAbsolutePath());

            } else {
                // Both formats of host key exist, we don't know which format
                // should be authoritative. Complain and abort.
                //
                stdKeys.add(objKey.getAbsolutePath());
                throw new Exception("Multiple host keys exist: " + stdKeys);
            }

        } else {
            if (stdKeys.isEmpty() || !SecurityUtils.isBouncyCastleRegistered()) {
                return new SimpleGeneratorHostKeyProvider();
            }
            return new FileKeyPairProvider(stdKeys.toArray(new String[stdKeys.size()]));
        }
    }

}
