/**
 * Copyright 2012 OW2 Shelbie
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.ow2.shelbie.commands.ssh.server.internal;

import org.apache.mina.util.Base64;
import org.apache.sshd.common.KeyPairProvider;
import org.apache.sshd.common.SshException;
import org.apache.sshd.common.util.Buffer;
import org.apache.sshd.server.PublickeyAuthenticator;
import org.apache.sshd.server.session.ServerSession;
import org.ow2.util.log.Log;
import org.ow2.util.log.LogFactory;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.security.KeyPair;
import java.security.PublicKey;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.Set;

/**
 * @author Loris Bouzonnet
 */
public class PublickeyAuthenticatorImpl implements PublickeyAuthenticator {

    private static Log log = LogFactory.getLog(PublickeyAuthenticatorImpl.class);

    private final Set<PublicKey> myHostKeys;

    private volatile PeerKeyCache peerKeyCache;

    private final HostKey hostKey;

    public PublickeyAuthenticatorImpl(final KeyPairProvider hostKeyProvider, final HostKey hostKey) {
        myHostKeys = myHostKeys(hostKeyProvider);
        this.hostKey = hostKey;
        peerKeyCache = new PeerKeyCache(hostKey.getPeerKeys());
    }

    private static Set<PublicKey> myHostKeys(KeyPairProvider p) {
        final Set<PublicKey> keys = new HashSet<PublicKey>(2);
        addPublicKey(keys, p, KeyPairProvider.SSH_RSA);
        addPublicKey(keys, p, KeyPairProvider.SSH_DSS);
        return keys;
    }

    private static void addPublicKey(final Collection<PublicKey> out,
                                     final KeyPairProvider p, final String type) {
        final KeyPair pair = p.loadKey(type);
        if (pair != null && pair.getPublic() != null) {
            out.add(pair.getPublic());
        }
    }

    public boolean authenticate(String username, PublicKey suppliedKey, ServerSession session) {

        if (username.equals(hostKey.getUser())) {
            if (myHostKeys.contains(suppliedKey)
                    || getPeerKeys().contains(suppliedKey)) {
                return true;
            }
        }
        return false;
    }

    private Set<PublicKey> getPeerKeys() {
        PeerKeyCache p = peerKeyCache;
        if (!p.isCurrent()) {
            p = p.reload();
            peerKeyCache = p;
        }
        return p.keys;
    }

    private static class PeerKeyCache {
        private final File path;
        private final long modified;
        final Set<PublicKey> keys;

        PeerKeyCache(final File path) {
            this.path = path;
            this.modified = path.lastModified();
            this.keys = read(path);
        }


        private static Set<PublicKey> read(File path) {
            try {
                final BufferedReader br = new BufferedReader(new FileReader(path));
                try {
                    final Set<PublicKey> keys = new HashSet<PublicKey>();
                    String line;
                    while ((line = br.readLine()) != null) {
                        line = line.trim();
                        if (line.startsWith("#") || line.length() == 0) {
                            continue;
                        }

                        String cleanLine;
                        if (line.startsWith("ssh-rsa ")) {
                            cleanLine = line.substring("ssh-rsa ".length());
                        } else {
                            cleanLine = line;
                        }
                        try {
                            byte[] bin = Base64.decodeBase64(cleanLine.getBytes());
                            keys.add(new Buffer(bin).getRawPublicKey());
                        } catch (RuntimeException e) {
                            logBadKey(path, cleanLine, e);
                        } catch (SshException e) {
                            logBadKey(path, cleanLine, e);
                        }
                    }
                    return Collections.unmodifiableSet(keys);
                } finally {
                    br.close();
                }
            } catch (FileNotFoundException noFile) {
                return Collections.emptySet();

            } catch (IOException err) {
                log.error("Cannot read " + path, err);
                return Collections.emptySet();
            }
        }

        private static void logBadKey(File path, String line, Exception e) {
            log.warn("Invalid key in " + path + ":\n  " + line, e);
        }

        boolean isCurrent() {
            return path.lastModified() == modified;
        }

        PeerKeyCache reload() {
            return new PeerKeyCache(path);
        }
    }
}
