/**
 * Copyright 2012 OW2 Shelbie
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.ow2.shelbie.commands.ssh.server.internal;

import java.io.Closeable;
import java.io.FilterOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.PrintStream;
import java.util.Map;

import jline.console.completer.Completer;
import org.apache.felix.service.command.CommandProcessor;
import org.apache.felix.service.command.CommandSession;
import org.apache.sshd.common.Factory;
import org.apache.sshd.server.Command;
import org.apache.sshd.server.Environment;
import org.apache.sshd.server.ExitCallback;
import org.ow2.shelbie.core.IPromptProvider;
import org.ow2.shelbie.core.console.JLineConsole;

public class ShellFactoryImpl implements Factory<Command>
{
    private final CommandProcessor commandProcessor;

    private final Completer completer;

    private JLineConsole console;

    private IPromptProvider promptProvider;

    public ShellFactoryImpl(final CommandProcessor commandProcessor, final Completer completer,
                            final IPromptProvider promptProvider) {
        this.commandProcessor = commandProcessor;
        this.completer = completer;
        this.promptProvider = promptProvider;
    }

    public Command create() {
        return new SimpleShell();
    }

    private class SimpleShell implements Command {

        private InputStream in;
        private OutputStream out;
        private OutputStream err;
        private CommandSession session;
        private ExitCallback callback;

        public void destroy() {
            // TODO Auto-generated method stub

        }

        public void setErrorStream(OutputStream err) {
            this.err = err;
        }

        public void setExitCallback(ExitCallback callback) {
            this.callback = callback;
        }

        public void setInputStream(InputStream in) {
            this.in = in;
        }

        public void setOutputStream(OutputStream out) {
            this.out = out;
        }

        public void start(Environment env) throws IOException {

            try {
                console = new JLineConsole(commandProcessor,
                        completer,
                        in,
                        new PrintStream(new LfToCrLfFilterOutputStream(out)),
                        new PrintStream(new LfToCrLfFilterOutputStream(err)),
                        promptProvider) {

                    @Override
                    public void run() {
                        super.run();
                        // the user has quit from the shell
                        // invoke a callback
                        onExit();
                    }

                };
                session = console.getSession();

                session.put("application.name", "shelbie");

                for (Map.Entry<String,String> e : env.getEnv().entrySet()) {
                    session.put(e.getKey(), e.getValue());
                }

                new Thread(console, "SSH Console").start();
            } catch (Exception e) {
                throw (IOException) new IOException("Unable to start shell").initCause(e);
            }

        }

        public void onExit() {
            console.close();
            // close the streams
            close(in, out, err);
            callback.onExit(0);
        }

    }

    private static void close(Closeable... closeables) {
        for (Closeable c : closeables) {
            try {
                c.close();
            } catch (IOException e) {
                // Ignore
            }
        }
    }

    public class LfToCrLfFilterOutputStream extends FilterOutputStream {

        private boolean lastWasCr;

        public LfToCrLfFilterOutputStream(OutputStream out) {
            super(out);
        }

        @Override
        public void write(int b) throws IOException {
            if (!lastWasCr && b == '\n') {
                out.write('\r');
                out.write('\n');
            } else {
                out.write(b);
            }
            lastWasCr = b == '\r';
        }

    }
}
