/**
 * Copyright 2012 OW2 Shelbie
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.ow2.shelbie.commands.ssh.server.internal;

import jline.console.completer.Completer;
import org.apache.felix.ipojo.annotations.Component;
import org.apache.felix.ipojo.annotations.Invalidate;
import org.apache.felix.ipojo.annotations.Property;
import org.apache.felix.ipojo.annotations.Requires;
import org.apache.felix.ipojo.annotations.Validate;
import org.apache.felix.service.command.CommandProcessor;
import org.apache.sshd.SshServer;
import org.apache.sshd.common.KeyPairProvider;
import org.apache.sshd.server.jaas.JaasPasswordAuthenticator;
import org.ow2.shelbie.commands.ssh.server.Constants;
import org.ow2.shelbie.commands.ssh.server.ISshDaemon;
import org.ow2.shelbie.core.IPromptProvider;

import java.security.Provider;
import java.security.Security;

@Component
public class SshDaemonComponent implements ISshDaemon {

	private SshServer server;

	@Requires
	private CommandProcessor provider;

    @Requires(filter = "(type=commands)")
    private Completer completer;

    @Requires(policy = "dynamic-priority")
    private IPromptProvider promptProvider;

	@Property(mandatory = true,
              name = Constants.SSHD_PORT)
	private int port;

    @Property
    private String passphrase;

	@Validate
	public void start() throws Exception {

        ClassLoader old = Thread.currentThread().getContextClassLoader();
        ClassLoader newCL = SshDaemonComponent.class.getClassLoader();
        Thread.currentThread().setContextClassLoader(newCL);
        try {
            boolean bcProv = true;
            try {
                Security.addProvider((Provider) newCL.loadClass(
                        "org.bouncycastle.jce.provider.BouncyCastleProvider").newInstance());
            } catch (ClassNotFoundException cnfe) {
                // BouncyCastle missing
                bcProv = false;
            }

            HostKey hostKey = HostKey.findHostKey();
            KeyPairProviderFactory keyPairProviderFactory = new KeyPairProviderFactory(hostKey, passphrase);
            KeyPairProvider keyPairProvider = keyPairProviderFactory.get();
            if (bcProv) {
                PasswordFinderSetter passwordFinderSetter = (PasswordFinderSetter)
                        newCL.loadClass("org.ow2.shelbie.commands.ssh.server.internal.SimplePasswordFinder").getConstructor(String.class).newInstance(passphrase);
                passwordFinderSetter.setPasswordFinder(keyPairProvider);
            }

            server = SshServer.setUpDefaultServer();
            server.setPort(port);
            server.setShellFactory(new ShellFactoryImpl(provider, completer, promptProvider));
            server.setCommandFactory(new CommandFactoryImpl(provider));
            server.setKeyPairProvider(keyPairProvider);
            if (hostKey.getDsaKey() != null || hostKey.getRsaKey() != null || hostKey.getObjKey() != null) {
                server.setPublickeyAuthenticator(new PublickeyAuthenticatorImpl(keyPairProvider, hostKey));
            }
            JaasPasswordAuthenticator pswdAuth = new JaasPasswordAuthenticator();
            pswdAuth.setDomain("shelbie");
            server.setPasswordAuthenticator(pswdAuth);
            server.start();
        } finally {
            Thread.currentThread().setContextClassLoader(old);
        }
	}

	@Invalidate
	public void stop() throws Exception {
		server.stop();
		server = null;
	}
}
