/*
 * Copyright 2007-2012 Bull S.A.S.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.ow2.spec.testengine;

import java.util.ArrayList;

import org.objectweb.asm.FieldVisitor;
import org.objectweb.asm.Opcodes;
import org.objectweb.asm.Type;
import org.objectweb.asm.commons.SerialVersionUIDAdder;

public class SerialVersionUIDReader extends SerialVersionUIDAdder {

    private AnalyzerClassVisitor analyzerClassVisitor;

    public SerialVersionUIDReader(final AnalyzerClassVisitor analyzerClassVisitor) {
        super(analyzerClassVisitor);
        this.analyzerClassVisitor = analyzerClassVisitor;
    }

    /**
     * Visits the header of the class.
     * @param version the class version.
     * @param access the class's access flags (see {@link Opcodes}). This
     *        parameter also indicates if the class is deprecated.
     * @param name the internal name of the class (see
     *        {@link Type#getInternalName() getInternalName}).
     * @param signature the signature of this class. May be <tt>null</tt> if
     *        the class is not a generic one, and does not extend or implement
     *        generic classes or interfaces.
     * @param superName the internal of name of the super class (see
     *        {@link Type#getInternalName() getInternalName}). For interfaces,
     *        the super class is {@link Object}. May be <tt>null</tt>, but
     *        only for the {@link Object} class.
     * @param interfaces the internal names of the class's interfaces (see
     *        {@link Type#getInternalName() getInternalName}). May be
     *        <tt>null</tt>.
     */
    @Override
    public void visit(int version, int access, String name, String signature, String superName, String[] interfaces) {
        this.name = name;
        // reset values
        computeSVUID = true;
        hasSVUID = false;
        hasStaticInitializer = false;
        svuidFields = new ArrayList();
        svuidConstructors = new ArrayList();
        svuidMethods = new ArrayList();
        super.visit(version, access, name, signature, superName, interfaces);
        // Not serializable
        if ("java/lang/Object".equals(superName) && (interfaces == null || interfaces.length == 0)) {
            computeSVUID = false;
        }
    }

    /*
     * Gets class field information for step 4 of the alogrithm. Also determines
     * if the class already has a SVUID.
     */
    @Override
    public FieldVisitor visitField(
            final int access,
            final String name,
            final String desc,
            final String signature,
            final Object value)
    {
        FieldVisitor fv = super.visitField(access, name, desc, signature, value);
        if (name.equals("serialVersionUID")) {
            // set the value
            analyzerClassVisitor.setUID((Long) value);
        }

        return fv;

    }


    @Override
    public void visitEnd() {
        // Set the id if it was computed
        if (computeSVUID) {
            try {
                analyzerClassVisitor.setUID(new Long(computeSVUID()));
            } catch (Exception e) {
                e.printStackTrace();
            }
        }

        cv.visitEnd();
    }

}
