/*
 * Copyright 2007-2012 Bull S.A.S.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.ow2.spec.testengine.xml;

import java.io.IOException;
import java.io.InputStreamReader;
import java.io.Reader;
import java.net.URL;
import java.net.URLConnection;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.w3c.dom.Document;
import org.xml.sax.EntityResolver;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

/**
 * Allows to parse an xml file.
 * @author Florent Benoit
 */
public final class DocumentParser {

    /**
     * Utility class.
     */
    private DocumentParser() {

    }

    /**
     * Builds a new Document for a given xml file.
     * @param url the URL of the the XML file.
     * @param isValidating validate or not the xml file ?
     * @param entityResolver the entityResolver used to validate document (if
     *        validating = true)
     * @throws DocumentParserException if creating of builder fails or parsing
     *         fails.
     * @return an application object.
     */
    public static Document getDocument(final URL url, final boolean isValidating, final EntityResolver entityResolver)
    throws DocumentParserException {
        // build factory
        DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();

        // XML files use schemas.
        factory.setNamespaceAware(true);
        factory.setValidating(isValidating);

        // ignore white space can only be set if parser is validating
        if (isValidating) {
            factory.setIgnoringElementContentWhitespace(true);
            factory.setAttribute("http://apache.org/xml/features/validation/schema", Boolean.valueOf(isValidating));
            factory.setAttribute("http://apache.org/xml/features/validation/schema-full-checking", Boolean.valueOf(true));
        }

        // Add schema location
        //TODO: FIXME: set other schemas ?
        if (isValidating) {
            factory.setAttribute("http://apache.org/xml/properties/schema/external-schemaLocation",
            "http://java.sun.com/xml/ns/persistence http://java.sun.com/xml/ns/persistence/persistence_1_0.xsd");
        }

        // Build a document builder
        DocumentBuilder builder = null;
        try {
            builder = factory.newDocumentBuilder();
        } catch (ParserConfigurationException e) {
            throw new DocumentParserException("Cannot build a document builder", e);
        }

        // Error handler (throwing exceptions)
        builder.setErrorHandler(new EasyBeansErrorHandler());

        // Dummy entity resolver if there is none
        if (entityResolver == null) {
            builder.setEntityResolver(new EmptyEntityResolver());
        } else {
            builder.setEntityResolver(entityResolver);
        }

        // Parse document
        URLConnection urlConnection = null;
        try {
            urlConnection = url.openConnection();
        } catch (IOException e) {
            throw new DocumentParserException("Cannot open a connection on URL '" + url + "'", e);
        }
        urlConnection.setDefaultUseCaches(false);
        Reader reader = null;
        try {
            reader = new InputStreamReader(urlConnection.getInputStream());
        } catch (IOException e) {
            throw new DocumentParserException("Cannot build an input stream reader on URL '" + url + "'", e);
        }

        InputSource inputSource = new InputSource(reader);
        Document document = null;
        try {
            document = builder.parse(inputSource);
        } catch (SAXException e) {
            throw new DocumentParserException("Cannot parse the XML file '" + url + "'.", e);
        } catch (IOException e) {
            throw new DocumentParserException("Cannot parse the XML file '" + url + "'.", e);
        } finally {
            // close InputStream when parsing is finished
            try {
                reader.close();
            } catch (IOException e) {
                throw new DocumentParserException("Cannot close the inputsource of the XML file'" + url + "'.", e);
            }
        }

        return document;
    }

}
