/*
 * Copyright 2007-2012 Bull S.A.S.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.ow2.spec.testengine.xml;


import java.io.IOException;
import java.net.URL;
import java.util.HashMap;
import java.util.Map;

import org.xml.sax.EntityResolver;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

/**
 * Entity resolver allowing to find schema within the classloader.
 * @author Florent Benoit
 */
public class SchemaEntityResolver implements EntityResolver {

    /**
     * Map where the schemas URLs are stored.
     */
    private Map<String, String> schemasUrls = null;

    /**
     * Constructor. Finds the XSD with classloader.
     * @param schemas the name of the schemas to resolve.
     */
    public SchemaEntityResolver(final String[] schemas) {
        schemasUrls = new HashMap<String, String>();
        URL url = null;
        for (int i = 0; i < schemas.length; i++) {
            url = SchemaEntityResolver.class.getResource("/" + schemas[i]);
            if (url == null) {
                throw new IllegalStateException("'" + schemas[i] + "' was not found in the current classloader !");
            }
            String urlString = url.toString();
            String id = urlString.substring(urlString.lastIndexOf('/') + 1);
            schemasUrls.put(id, urlString);

        }
    }

    /**
     * The Parser will call this method before opening any external entity
     * except the top-level document entity.
     * @param publicId The public identifier of the external entity being
     *        referenced, or null if none was supplied.
     * @param systemId The system identifier of the external entity being
     *        referenced.
     * @return An InputSource object describing the new input source, or null to
     *         request that the parser open a regular URI connection to the
     *         system identifier.
     * @throws SAXException Any SAX exception, possibly wrapping another
     *         exception.
     * @throws IOException A Java-specific IO exception, possibly the result of
     *         creating a new InputStream or Reader for the InputSource.
     */
    public InputSource resolveEntity(final String publicId, final String systemId) throws IOException, SAXException {

        String localPath = null;

        if (systemId != null) {
            // Can be a schema
            if (systemId.toLowerCase().endsWith(".xsd")) {
                // Retrieve basename
                String baseName = systemId.substring(systemId.lastIndexOf('/') + 1);

                // Registred ?
                localPath = schemasUrls.get(baseName);
            }
        }

        // schema not found
        if (localPath == null) {
            throw new SAXException("No XSD found for '" + systemId + "'.");
        }

        // Return the local path source
        return (new InputSource(localPath));
    }

}
