/**
 * EasyBeans
 * Copyright (C) 2011 Bull S.A.S.
 * Contact: easybeans@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: DataSourceDefinition.java 5871 2011-05-05 12:11:35Z benoitf $
 * --------------------------------------------------------------------------
 */

package javax.annotation.sql;

import static java.lang.annotation.ElementType.TYPE;
import static java.lang.annotation.RetentionPolicy.RUNTIME;

import java.lang.annotation.Retention;
import java.lang.annotation.Target;

/**
 * This annotation is used to define a container DataSource and be registered with JNDI. The DataSource may be configured by
 * setting the annotation elements for commonly used DataSource properties. Additional standard and vendor-specific properties may
 * be specified using the properties element. The data source will be registered under the name specified in the name element. It
 * may be defined to be in any valid Java EE namespace, and will determine the accessibility of the data source from other
 * components. A JDBC driver implementation class of the appropriate type, either DataSource, ConnectionPoolDataSource, or
 * XADataSource, must be indicated by the className element. The driver class is not required to be available at deployment but
 * must be available at runtime prior to any attempt to access the DataSource .
 * @see <a href="http://jcp.org/en/jsr/detail?id=250">Annotation 1.1 specification</a>
 * @author Florent Benoit
 * @since Annotation 1.1 version.
 */
@Target( { TYPE })
@Retention(RUNTIME)
public @interface DataSourceDefinition {

    /**
     * JNDI name by which the data source name will be registered.
     */
    String name();

    /**
     * DataSource implementation class className.
     */
    String className();

    /**
     * Description of the data source.
     */
    String description() default "";

    /**
     * A JDBC url. If the url property is specified along with other standard DataSource properties such as serverName and
     * portNumber, the more specific properties will take precedence and the url will be ignored
     */
    String url() default "";

    /**
     * User name for connection authentications
     */
    String user() default "";

    /**
     * password for connection authentications
     */
    String password() default "";

    /**
     * Name of a database on a server
     */
    String databaseName() default "";

    /**
     * Port number where a server is listening for requests
     */
    int portNumber() default -1;

    /**
     * Database server name
     */
    String serverName() default "localhost";

    /**
     * Isolation level for connections.
     */
    int isolationLevel() default -1;

    /**
     * Indicates whether a connection is transactional or not.
     */
    boolean transactional() default true;

    /**
     * Number of connections that should be created when a connection pool is initialized
     */
    int initialPoolSize() default -1;

    /**
     * Maximum number of connections that should be concurrently allocated for a connection pool
     */
    int maxPoolSize() default -1;

    /**
     * Minimum number of connections that should be allocated for a connection pool
     */
    int minPoolSize() default -1;

    /**
     * The number of seconds that a physical connection should remain unused in the pool before the connection is closed for a
     * connection pool
     */
    int maxIdleTime() default -1;

    /**
     * The total number of statements that a connection pool should keep open. A value of 0 indicates that the caching of
     * statements is disabled for a connection pool.
     */
    int maxStatements() default -1;

    /**
     * Used to specify vendor specific properties and less commonly used DataSource properties.
     */
    String[] properties() default {};

    /**
     * The maximum time in seconds that this data source will wait while attempting to connect to a database. A value of 0
     * specifies that the timeout is the default system timeout if there is one, otherwise it specifies that there is no timeout
     */
    int loginTimeout() default 0;
}
