/**
 * OW2 Spec
 * Copyright (C) 2010 Bull S.A.S.
 * Contact: easybeans@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: Event.java 5375 2010-02-25 17:25:09Z sauthieg $
 * --------------------------------------------------------------------------
 */

package javax.enterprise.event;

import java.lang.annotation.Annotation;

import javax.enterprise.util.TypeLiteral;

/**
 * Allows the application to fire events of a particular type.
 * Beans fire events via an instance of the Event interface, which may be injected:
 * <pre>
 *  &#064;Inject @Any Event&lt;LoggedInEvent> loggedInEvent;
 * </pre>
 *
 * The {@code fire()} method accepts an event object:
 * <pre>
 *  public void login() {
 *    ...
 *    loggedInEvent.fire( new LoggedInEvent(user) );
 *  }
 * </pre>
 *
 * Any combination of qualifiers may be specified at the injection point:
 * <pre>
 *  &#064;Inject @Admin Event&lt;LoggedInEvent> adminLoggedInEvent;
 * </pre>
 *
 * Or, the @{@link javax.enterprise.inject.Any} qualifier may be used, allowing the
 * application to specify qualifiers dynamically:
 * <pre>
 *  &#064;Inject @Any Event&lt;LoggedInEvent> loggedInEvent;
 * </pre>
 *
 * For an injected Event:
 * <ul>
 *  <li>the specified type is the type parameter specified at the injection point, and</li>
 *  <li>the specified qualifiers are the qualifiers specified at the injection point.</li>
 * </ul>
 *
 * @author Guillaume Sauthier
 */
public interface Event<T> {

    /**
     * Fires an event with the specified qualifiers and notifies observers.
     * 
     * @param event the event object
     * @throws IllegalArgumentException if the runtime type of the event object contains
     *         a type variable
     */
    void fire(T event);

    /**
     * Obtains a child Event for the given additional required qualifiers.
     *
     * @param qualifiers the additional specified qualifiers 
     * @return the child Event
     * @throws IllegalArgumentException if passed two instances of the same qualifier type,
     *         or an instance of an annotation that is not a qualifier type
     */
    Event<T> select(Annotation... qualifiers);

    /**
     * Obtains a child Event for the given required type and additional required qualifiers.
     * 
     * @param subtype a Class representing the specified type
     * @param qualifiers the additional specified qualifiers 
     * @param <U> the specified type
     * @return the child Event
     * @throws IllegalArgumentException if passed two instances of the same qualifier type,
     *         or an instance of an annotation that is not a qualifier type 
     */
    <U extends T> Event<U> select(Class<U> subtype,
                                  Annotation... qualifiers);

    /**
     * Obtains a child Event for the given required type and additional required qualifiers.
     * 
     * @param subtype a {@link javax.enterprise.util.TypeLiteral} representing the specified type
     * @param qualifiers the additional specified qualifiers 
     * @param <U> the specified type
     * @return the child Event
     * @throws IllegalArgumentException if passed two instances of the same qualifier type, or an instance of an annotation that is not a qualifier type
     */
    <U extends T> Event<U> select(TypeLiteral<U> subtype,
                                  Annotation... qualifiers);
    
}
