/**
 * OW2 Spec
 * Copyright (C) 2010 Bull S.A.S.
 * Contact: easybeans@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: Observes.java 5375 2010-02-25 17:25:09Z sauthieg $
 * --------------------------------------------------------------------------
 */

package javax.enterprise.event;

import java.lang.annotation.Documented;
import java.lang.annotation.Retention;
import java.lang.annotation.Target;

import static java.lang.annotation.ElementType.PARAMETER;
import static java.lang.annotation.RetentionPolicy.RUNTIME;

/**
 * Identifies the event parameter of an observer method. May be applied to a parameter of a
 * method of a bean class or extension.
 * <pre>
 *  public void afterLogin(@Observes LoggedInEvent event) { ... }
 * </pre>
 *
 * An observer method is a non-abstract method of a managed bean class or session bean class
 * (or of an extension). An observer method may be either static or non-static. If the bean
 * is a session bean, the observer method must be either a business method of the EJB or a
 * static method of the bean class.
 *
 * Each observer method must have exactly one event parameter, of the same type as the event
 * type it observes. Event qualifiers may be declared by annotating the event parameter. When
 * searching for observer methods for an event, the container considers the type and qualifiers
 * of the event parameter.
 *
 * If the event parameter does not explicitly declare any qualifier, the observer method observes
 * events with no qualifier.
 *
 * The event parameter type may contain a type variable or wildcard.
 *
 * A bean (or extension) may declare multiple observer methods.
 *
 * Observer methods are inherited by bean subclasses.
 *
 * Interceptors and decorators may not declare observer methods.
 
 * @author Guillaume Sauthier
 *
 * @since CDI 1.0
 */
@Target(PARAMETER)
@Retention(RUNTIME)
@Documented
public @interface Observes {

    /**
     * Specifies under what conditions the observer method is notified.
     *
     * By default, the observer method is notified even if no instance of the bean that defines
     * the observer method already exists in the current context.
     */
    Reception notifyObserver() default Reception.ALWAYS;

    /**
     * Specifies at what time the observer method is notified.
     *
     * By default, the observer method is notified when the event is fired.
     */
    TransactionPhase during() default TransactionPhase.IN_PROGRESS;
}