/**
 * OW2 Specifications
 * Copyright (C) 2007 Bull S.A.S.
 * Contact: easybeans@objectweb.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: ExecutionContext.java 4359 2008-12-10 10:24:40Z sauthieg $
 * --------------------------------------------------------------------------
 */

package javax.resource.spi.work;

import javax.resource.NotSupportedException;
import javax.transaction.xa.Xid;

/**
 * This class models an execution context (transaction, security, etc) with
 * which the {@link Work} instance must be executed. This class is provided
 * as a convenience for easily creating {@link ExecutionContext} instances
 * by extending this class and overriding only those methods of interest.<br/>
 *
 * Some reasons why it is better for ExecutionContext to be a class rather
 * than an interface:<br/>
 * <ul>
 *   <li>There is no need for a resource adapter to implement this class.
 *   It only needs to implement the context information like transaction, etc.</li>
 *   <li>The resource adapter code does not have to change when the
 *   ExecutionContext class evolves. For example, more context types could
 *   be added to the ExecutionContext class (in the future) without forcing
 *   resource adapter implementations to change.</li>
 * </ul>
 * @version 1.0
 * @author Guillaume Sauthier
 */
public class ExecutionContext {

    /**
     * Transaction Timeout.
     */
    private long transactionTimeout;

    /**
     * Transaction context.
     */
    private Xid xid;

    /**
     * Constructs a new ExceutionContext with default values. Using
     * <code>WorkManager.UNKNOWN</code> as default timeout value.
     */
    public ExecutionContext() {
        transactionTimeout = WorkManager.UNKNOWN;
    }

    /**
     * Set a transaction context.
     * @param xid transaction context.
     */
    public void setXid(Xid xid) {
        this.xid = xid;
    }

    /**
     * Returns the transaction context.
     * @return the transaction context.
     */
    public Xid getXid(){
        return xid;
    }

    /**
     * Set the transaction timeout value for a imported transaction.
     * @param timeout transaction timeout value in seconds. Only positive
     *        non-zero values are accepted. Other values are illegal and
     *        are rejected with a NotSupportedException.
     * @throws NotSupportedException thrown to indicate an illegal timeout
     *         value.
     */
    public void setTransactionTimeout(long timeout) throws NotSupportedException {
        if (timeout <= 0) {
            throw new NotSupportedException("Transaction timeout must be positive and non-zero ("
                                            + timeout + ")");
        }
        this.transactionTimeout = timeout;
    }

    /**
     * Get the transaction timeout value for a imported transaction.
     * @return the specified transaction timeout value in seconds. When no
     *         timeout value or an illegal timeout value had been specified,
     *         a value of -1 (WorkManager.UNKNOWN) is returned; such a
     *         transaction is excluded from regular timeout processing.
     */
    public long getTransactionTimeout() {
        return this.transactionTimeout;
    }
}
