/**
 * OW2 Specifications
 * Copyright (C) 2007 Bull S.A.S.
 * Contact: easybeans@objectweb.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: BootstrapContext.java 6201 2012-03-21 10:28:10Z benoitf $
 * --------------------------------------------------------------------------
 */
 
package javax.resource.spi;

import java.util.Timer;

import javax.resource.spi.work.WorkContext;
import javax.resource.spi.work.WorkManager;
import javax.transaction.TransactionSynchronizationRegistry;

/**
 * This provides a mechanism to pass a bootstrap context to a
 * resource adapter instance when it is bootstrapped. That is,
 * when (start(BootstrapContext)) method on the ResourceAdapter
 * class is invoked. The bootstrap context contains references
 * to useful facilities that could be used by the resource
 * adapter instance.
 * @version 1.0
 * @author Guillaume Sauthier
 */
public interface BootstrapContext {

    /**
     * Provides a handle to a WorkManager instance. The WorkManager
     * instance could be used by a resource adapter to do its work
     * by submitting Work instances for execution.
     * @return a WorkManager instance.
     */
    WorkManager getWorkManager();
    
    /**
     * Provides a handle to a XATerminator instance. The XATerminator
     * instance could be used by a resource adapter to flow-in
     * transaction completion and crash recovery calls from an EIS.
     * @return a XATerminator instance.
     */
    XATerminator getXATerminator();
    
    /**
     * Creates a new java.util.Timer instance. The Timer instance
     * could be used to perform periodic Work executions or other tasks.
     * @return a new Timer instance.
     * @throws UnavailableException indicates that a Timer instance is
     *         not available. The request may be retried later.
     */
    Timer createTimer()throws UnavailableException;
    
    /**
     * A resource adapter can check an application server's support for a particular WorkContext type through this method. This mechanism enables a resource adapter developer to dynamically change the WorkContexts submitted with a Work instance based on the support provided by the application server. The application server must employ an exact type equality check (that is java.lang.Class.equals(java.lang.Class) check) in this method, to check if it supports the WorkContext type provided by the resource adapter. This method must be idempotent, that is all calls to this method by a resource adapter for a particular WorkContext type must return the same boolean value throughout the lifecycle of that resource adapter instance. 
     * @param workContextClass The WorkContext type that is tested for support by the application server. 
     * @return true if the workContextClass is supported by the application server. false if the workContextClass is unsupported or unknown to the application server.
     * @since 1.6 
     */
    boolean isContextSupported(java.lang.Class<? extends WorkContext> workContextClass);
    
    /**
     * Provides a handle to a TransactionSynchronization instance. The TransactionSynchronizationRegistry instance could be used by a resource adapter to register synchronization objects, get transaction state and status etc. This interface is implemented by the application server by a stateless service object. The same object can be used by any number of resource adapter objects with thread safety. 
     * @return a TransactionSynchronizationRegistry instance.
     * @since 1.6
     */
    TransactionSynchronizationRegistry getTransactionSynchronizationRegistry();
    
}
