/**
 * OW2 Specifications
 * Copyright (C) 2007 Bull S.A.S.
 * Contact: easybeans@objectweb.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: PasswordCredential.java 6201 2012-03-21 10:28:10Z benoitf $
 * --------------------------------------------------------------------------
 */

package javax.resource.spi.security;

import java.io.Serializable;
import java.util.Arrays;

import javax.resource.spi.ManagedConnectionFactory;

/**
 * The class PasswordCredential acts as a holder for username and password.
 * @since 0.6
 * @version 0.6
 * @author Guillaume Sauthier
 */
public final class PasswordCredential implements Serializable {

    /**
     * Serial version UID.
     */
    private static final long serialVersionUID = -1770833344350711674L;

    /**
     * User name.
     */
    private String userName;

    /**
     * Password.
     */
    private char[] password;

    /**
     * {@link ManagedConnectionFactory} instance.
     */
    private ManagedConnectionFactory managedConnectionFactory;

    /**
     * Creates a new PasswordCredential object from the given user name
     * and password.<br/>
     *
     * Note that the given user password is cloned before it is stored
     * in the new PasswordCredential object.
     * @param userName
     * @param password
     */
    public PasswordCredential(final String userName,
                              final char[] password) {
        this.userName = userName;
        this.password = password.clone();
    }

    /**
     * Returns the user name.
     * @return the userName
     */
    public String getUserName() {
        return userName;
    }

    /**
     * Returns the user password.<br/>
     *
     * Note that this method returns a reference to the password. It
     * is the caller's responsibility to zero out the password
     * information after it is no longer needed.
     * @return the password
     */
    public char[] getPassword() {
        return password;
    }

    /**
     * Gets the target ManagedConnectionFactory for which the user name
     * and password has been set by the application server. A
     * {@link ManagedConnectionFactory} uses this field to find out
     * whether PasswordCredential should be used by it for sign-on to
     * the target EIS instance.
     * @return ManagedConnectionFactory instance for which user name and
     *         password have been specified
     */
    public ManagedConnectionFactory getManagedConnectionFactory() {
        return this.managedConnectionFactory;
    }

    /**
     * Sets the target ManagedConenctionFactory instance for which the
     * user name and password has been set by the application server.
     * @param mcf ManagedConnectionFactory instance for which user name
     *        and password have been specified
     */
    public void setManagedConnectionFactory(ManagedConnectionFactory mcf) {
        this.managedConnectionFactory = mcf;
    }

    /**
     * Compares this PasswordCredential with the specified object for
     * equality. The two PasswordCredential instances are the same if
     * they are equal in username and password.
     * @param other Object to which PasswordCredential is to be compared
     * @return if and if the specified object is a PasswordCredential
     *         whose username and password are equal to this instance.
     */
    public boolean equals(Object other) {
        if (!(other instanceof PasswordCredential)) {
            return false;
        }

        PasswordCredential credential = (PasswordCredential) other;
        if (!userName.equals(credential.userName)) {
            return false;
        }
        if (!Arrays.equals(password, credential.password)) {
            return false;
        }

        return true;

    }

    /**
     * Returns the hash code for this PasswordCredential
     * @return hash code for this PasswordCredential
     */
    public int hashCode() {
        return userName.hashCode() + Arrays.hashCode(password);
    }
}
