/**
 * OW2 Specifications
 * Copyright (C) 2007 Bull S.A.S.
 * Contact: easybeans@objectweb.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: DeploymentConfiguration.java 4359 2008-12-10 10:24:40Z sauthieg $
 * --------------------------------------------------------------------------
 */

package javax.enterprise.deploy.spi;

import java.io.InputStream;
import java.io.OutputStream;

import javax.enterprise.deploy.model.DDBeanRoot;
import javax.enterprise.deploy.model.DeployableObject;
import javax.enterprise.deploy.spi.exceptions.BeanNotFoundException;
import javax.enterprise.deploy.spi.exceptions.ConfigurationException;

/**
 * An interface that defines a container for all the server-specific configuration
 * information for a single top-level J2EE module. The DeploymentConfiguration
 * object could represent a single stand alone module or an EAR file that contains
 * several sub-modules.
 * @author Guillaume Sauthier
 */
public interface DeploymentConfiguration {

    /**
     * Returns an object that provides access to the deployment descriptor data
     * and classes of a J2EE module.
     * @return DeployableObject
     */
    DeployableObject getDeployableObject();

    /**
     * Returns the top level configuration bean, DConfigBeanRoot, associated with
     * the deployment descriptor represented by the designated DDBeanRoot bean.
     * @param bean The top level bean that represents the associated deployment
     *        descriptor.
     * @return the DConfigBeanRoot for editing the server-specific properties
     *         required by the module.
     * @throws ConfigurationException reports errors in generating a configuration bean
     */
    DConfigBeanRoot getDConfigBeanRoot(DDBeanRoot bean)
                    throws ConfigurationException;

    /**
     * Remove the root DConfigBean and all its children.
     * @param bean the top leve DConfigBean to remove.
     * @throws BeanNotFoundException the bean provided is not in this beans child list.
     */
    void removeDConfigBean(DConfigBeanRoot bean) throws BeanNotFoundException;

    /**
     * Restore from disk to instantated objects all the DConfigBeans associated
     * with a specific deployment descriptor. The beans may be fully or partially
     * configured.
     * @param inputArchive The input stream for the file from which the DConfigBeans
     *        should be restored.
     * @param bean The DDBeanRoot bean associated with the deployment descriptor file.
     * @return The top most parent configuration bean, DConfigBeanRoot
     * @throws ConfigurationException reports errors in generating a configuration bean
     */
    DConfigBeanRoot restoreDConfigBean(InputStream inputArchive,
                                       DDBeanRoot bean)
                    throws ConfigurationException;

    /**
     * Save to disk all the configuration beans associated with a particular
     * deployment descriptor file. The saved data may be fully or partially
     * configured DConfigBeans. The output file format is recommended to be XML.
     * @param outputArchive The output stream to which the DConfigBeans should
     *        be saved.
     * @param bean The top level bean, DConfigBeanRoot, from which to be save.
     * @throws ConfigurationException reports errors in saving a configuration bean
     */
    void saveDConfigBean(OutputStream outputArchive,
                         DConfigBeanRoot bean) throws ConfigurationException;

    /**
     * Restore from disk to a full set of configuration beans previously stored.
     * @param inputArchive The input stream from which to restore the Configuration.
     * @throws ConfigurationException reports errors in loading a configuration bean
     */
    void restore(InputStream inputArchive) throws ConfigurationException;

    /**
     * Save to disk the current set configuration beans created for this deployable
     * module. It is recommended the file format be XML.
     * @param outputArchive The output stream to which to save the Configuration.
     * @throws ConfigurationException
     */
    void save(OutputStream outputArchive) throws ConfigurationException;
}
