/**
 * EasyBeans
 * Copyright (C) 2010 Bull S.A.S.
 * Contact: easybeans@objectweb.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: PasswordValidationCallback.java 6201 2012-03-21 10:28:10Z benoitf $
 * --------------------------------------------------------------------------
 */
package javax.security.auth.message.callback;

import java.util.Arrays;
import javax.security.auth.Subject;

/**
 * Callback for PasswordValidation.
 * This callback may be used by an authentication module
 * to employ the password validation facilities of its containing runtime.
 * This Callback would typically be called by a <code>ServerAuthModule</code>
 * during <code>validateRequest</code> processing.
 *
 * @version 1.0
 */
public class PasswordValidationCallback implements javax.security.auth.callback.Callback {
    private static final long serialVersionUID = -135437753287939796L;

    /**
     * The subject to authenticate, read by the callback handler.
     */
    private Subject theSubject;

    /**
     * The username to authenticate, read by the callback handler.
     */
    private String theUsername;

    /**
     * The password to use for authentication, read by the callback handler.
     */
    private char[] thePassword;

    /**
     * The authentication result, SET by the callback handler.
     */
    private boolean theResult;


    /**
     * Create a PasswordValidationCallback.
     *
     * @param subject The subject for authentication
     *
     * @param username The username to authenticate
     *
     * @param password The user's password, which may be null.
     */
    public PasswordValidationCallback(Subject subject, String username,
            char[] password) {
        this.theSubject = subject;
        this.theUsername = username;
        if (password != null) {
            this.thePassword = (char[]) password.clone();
        }
    }

    /**
     * Get the subject.
     *
     * @return The subject.
     */
    public Subject getSubject() {
        return this.theSubject;
    }

    /**
     * Get the username.
     *
     * @return The username.
     */
    public String getUsername() {
	return this.theUsername;
    }

    /**
     * Get the password.
     *
     * <p />
     *
     * Note that this method returns a reference to the password.
     * If a clone of the array is created it is the caller's
     * responsibility to zero out the password information after
     * it is no longer needed.
     *
     * @return The password, which may be null.
     */
    public char[] getPassword() {
	return this.thePassword;
    }

    /**
     * Clear the password.
     */
    public void clearPassword() {
	if (this.thePassword != null) {
	    Arrays.fill(this.thePassword, ' ');
	}
    }

    /**
     * Set the authentication result.
     *
     * @param result True if authentication succeeded, false otherwise
     */
    public void setResult(boolean result) {
	this.theResult = result;
    }

    /**
     * Get the authentication result.
     *
     * @return True if authentication succeeded, false otherwise
     */
    public boolean getResult() {
	return this.theResult;
    }

}
