/**
 * EasyBeans
 * Copyright (C) 2010 Bull S.A.S.
 * Contact: easybeans@objectweb.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: PrivateKeyCallback.java 6201 2012-03-21 10:28:10Z benoitf $
 * --------------------------------------------------------------------------
 */
package javax.security.auth.message.callback;

import java.math.BigInteger;
import java.security.PrivateKey;
import java.security.cert.Certificate;
import javax.security.auth.x500.X500Principal;

/**
 * Callback for acquiring a Public Key Infrastructure (PKI) private key
 * and its corresponding certificate chain.
 * This Callback may be used by client or server authentication modules
 * to obtain private keys or private key references, from key
 * repositories available to the CallbackHandler that processes the Callback.
 *
 * @version 1.0
 */
public class PrivateKeyCallback implements javax.security.auth.callback.Callback {
    private static final long serialVersionUID = 112472314619902246L;

    /**
     * The request, read by the callback handler to find the private key.
     */
    private Request theRequest;
    /**
     * The private key, SET by the callback handler.
     */
    private PrivateKey thePrivateKey;
    /**
     * The certificate chain, SET by the callback handler.
     */
    private Certificate[] theCertificateChain;

    /**
     * Marker interface for private key request types.
     */
    public static interface Request {
    };

    /**
     * Request type for private keys that are identified using an alias.
     */
    public static class AliasRequest implements Request {
        private static final long serialVersionUID = 9172666777472571517L;

        /**
         * The alias, identifying a private key.
         */
        private String theAlias;

        /**
         * Construct an AliasRequest with an alias.
         *
         * <p />
         *
         * The alias is used to directly identify the private key
         * to be returned.  The corresponding certificate chain for the
         * private key is also returned.
         *
         * <p />
         *
         * If the alias is null,
         * the handler of the callback relies on its own default.
         *
         * @param alias Name identifier for the private key, or null.
         */
        public AliasRequest(String alias) {
            this.theAlias = alias;
        }

        /**
         * Get the alias.
         *
         * @return The alias, or null.
         */
        public String getAlias() {
            return this.theAlias;
        }
    }

    /**
     * Request type for private keys that are identified using a SubjectKeyID.
     */
    public static class SubjectKeyIDRequest implements Request {
        private static final long serialVersionUID = -8785738423304179884L;

        /**
         * The subject key ID, identifying a private key.
         */
        private byte[] theSubjectKeyID;

        /**
         * Construct a SubjectKeyIDRequest with an subjectKeyID.
         *
         * <p />
         *
         * The subjectKeyID is used to directly identify the private key
         * to be returned.  The corresponding certificate chain for the
         * private key is also returned.
         *
         * <p />
         *
         * If the subjectKeyID is null,
         * the handler of the callback relies on its own default.
         *
         * @param subjectKeyID Identifier for the private key, or null.
         */
        public SubjectKeyIDRequest(byte[] subjectKeyID) {
            if (subjectKeyID != null) {
                this.theSubjectKeyID = (byte[]) subjectKeyID.clone();
            }
        }

        /**
         * Get the subjectKeyID.
         *
         * @return The subjectKeyID, or null.
         */
        public byte[] getSubjectKeyID() {
            return this.theSubjectKeyID;
        }
    }

    /**
     * Request type for private keys that are identified using an
     * issuer/serial number.
     */
    public static class IssuerSerialNumRequest implements Request {
        private static final long serialVersionUID = 6590696146248952744L;

        /**
         * The issuer of the requested private key.
         */
        private X500Principal theIssuer;
        /**
         * The serial number of the requested private key.
         */
        private BigInteger theSerialNumber;

        /**
         * Constructs a IssuerSerialNumRequest with an issuer/serial number.
         *
         * <p />
         *
         * The issuer/serial number is used to identify a
         * public key certificate.  The corresponding private key
         * is returned in the callback.  The corresponding certificate chain
         * for the private key is also returned.
         *
         * If the issuer/serialNumber parameters are null,
         * the handler of the callback relies on its own defaults.
         *
         * @param issuer The X500Principal name of the certificate issuer,
         *		or null.
         *
         * @param serialNumber The serial number of the certificate,
         *		or null.
         */
        public IssuerSerialNumRequest(X500Principal issuer,
                BigInteger serialNumber) {
            this.theIssuer = issuer;
            this.theSerialNumber = serialNumber;
        }

        /**
         * Get the issuer.
         *
         * @return The issuer, or null.
         */
        public X500Principal getIssuer() {
            return this.theIssuer;
        }

        /**
         * Get the serial number.
         *
         * @return The serial number, or null.
         */
        public BigInteger getSerialNum() {
            return this.theSerialNumber;
        }
    }

    /**
     * Request type for private keys that are identified using a
     * certificate digest or thumbprint.
     */
    public static class DigestRequest implements Request {
        private static final long serialVersionUID = -2159280822935687894L;

        /**
         * The digest, identifying the requested private key.
         */
        private byte[] theDigest;

        /**
         * The digest algorithm (null or conforming to the MessageDigest
         * algorithm parameter).
         */
        private String theAlgorithm;

        /**
         * Constructs a DigestRequest with a digest value and algorithm
         * identifier.
         *
         * <p />
         *
         * The digest of the certificate whose private key is returned
         * must match the provided digest. The certificate digest is computed
         * by applying the specified algorithm to the bytes of the certificate.
         * For example:
         * <code>
         * MessageDigest.getInstance(algorithm).digest(cert.getEncoded())
         * </code>. The corresponding certificate chain for the private key is
         * also returned.
         *
         * If the digest or algorithm parameters are null,
         * the handler of the callback relies on its own defaults.
         *
         * @param digest The digest value to use to select the corresponding
         * certificate and private key (or null).
         *
         * @param algorithm A string value identifying the digest algorithm.
         * The value passed to this parameter may be null. If it is not null,
         * it must conform to the requirements for the algorithm parameter of
         * <code>java.security.MessageDigest.getInstance()</code>.
         */
        public DigestRequest(byte[] digest, String algorithm) {
            if (digest != null) {
                this.theDigest = (byte[]) digest.clone();
            }
            this.theAlgorithm = algorithm;
        }

        /**
         * Get the digest value.
         *
         * @return The digest value which must match the digest of the
         * certificate corresponding to the returned private key.
         */
        public byte[] getDigest() {
            return this.theDigest;
        }

        /**
         * Get the algorithm identifier.
         *
         * @return The identifer of the algorithm used to compute the digest.
         */
        public String getAlgorithm() {
            return this.theAlgorithm;
        }
    }

    /**
     * Constructs this PrivateKeyCallback with a private key Request object.
     *
     * <p />
     *
     * The <i>request</i> object identifies the private key
     * to be returned.  The corresponding certificate chain for the
     * private key is also returned.
     *
     * <p />
     *
     * If the <i>request</i> object is null,
     * the handler of the callback relies on its own default.
     *
     * @param request Identifier for the private key, or null.
     */
    public PrivateKeyCallback(Request request) {
        this.theRequest = request;
    }

    /**
     * Used by the CallbackHandler to
     * get the Request object that identifies the private key to be returned.
     *
     * @return The Request object which identifies the private key
     *		to be returned, or null.  If null, the handler of the callback
     *		relies on its own default.
     */
    public Request getRequest() {
        return this.theRequest;
    }

    /**
     * Used by the CallbackHandler to set the requested private key and
     * the corresponding certificate chain within the Callback.
     *
     * <p />
     *
     * If the requested private key or chain could not be found,
     * then both values must be set to null.
     *
     * @param key The private key, or null.
     *
     * @param chain The corresponding certificate chain, or null.
     */
    public void setKey(PrivateKey key, Certificate[] chain) {
        this.thePrivateKey = key;
        if (chain != null) {
            this.theCertificateChain = (Certificate[]) chain.clone();
        } else {
            this.theCertificateChain = null;
        }
    }

    /**
     * Used to obtain the private key set within the Callback.
     *
     * @return The private key, or null if the key could not be found.
     */
    public PrivateKey getKey() {
        return this.thePrivateKey;
    }

    /**
     * Used to obtain the certificate chain set within the Callback.
     *
     * @return The certificate chain, or null if the chain could not be found.
     */
    public Certificate[] getChain() {
        return this.theCertificateChain;
    }
}
