/**
 * EasyBeans
 * Copyright (C) 2010 Bull S.A.S.
 * Contact: easybeans@objectweb.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: SecretKeyCallback.java 6201 2012-03-21 10:28:10Z benoitf $
 * --------------------------------------------------------------------------
 */
package javax.security.auth.message.callback;

import javax.crypto.SecretKey;

/**
 * Callback for acquiring a shared secret from a key repository.
 * This Callback may be used by client or server authentication modules
 * to obtain shared secrets (for example, passwords) without relying on a
 * user during the Callback processing.
 * This Callback is typically employed by <code>ClientAuthModules</code>
 * invoked from intermediate components that need to acquire
 * a password to authenticate to their target service.
 *
 * @version 1.0
 */
public class SecretKeyCallback implements javax.security.auth.callback.Callback {
    private static final long serialVersionUID = -5850560716004066432L;

    /**
     * The request, read by the callback handler to find the secret key.
     */
    private Request theRequest;
    /**
     * The secret key, set by the callback handler.
     */
    private SecretKey theSecretKey;

    /**
     * Marker interface for secret key request types.
     */
    public static interface Request {
    };

    /**
     * Request type for secret keys that are identified using an alias.
     */
    public static class AliasRequest implements Request {
        private static final long serialVersionUID = -7053724641214627621L;

        /**
         * The alias, identifying a secret key.
         */
        private String theAlias;

        /**
         * Construct an AliasRequest with an alias.
         *
         * <p />
         *
         * The alias is used to directly identify the secret key
         * to be returned.
         *
         * <p />
         *
         * If the alias is null,
         * the handler of the callback relies on its own default.
         *
         * @param alias Name identifier for the secret key, or null.
         */
        public AliasRequest(String alias) {
            this.theAlias = alias;
        }

        /**
         * Get the alias.
         *
         * @return The alias, or null.
         */
        public String getAlias() {
            return this.theAlias;
        }
    }

    /**
     * Constructs this SecretKeyCallback with a secret key Request object.
     *
     * <p />
     *
     * The <i>request</i> object identifies the secret key
     * to be returned.
     *
     * If the alias is null, the handler of the callback
     * relies on its own default.
     *
     * @param request Request object identifying the secret key, or null.
     */
    public SecretKeyCallback(Request request) {
        this.theRequest = request;
    }

    /**
     * Used by the CallbackHandler to get the Request object which identifies
     * the secret key to be returned.
     *
     * @return The Request object which identifies the private key
     *		to be returned, or null.  If null, the handler of the callback
     *		relies on its own default.
     */
    public Request getRequest() {
        return this.theRequest;
    }

    /**
     * Used by the CallbackHandler to set the requested secret key within
     * the Callback.
     *
     * @param key The secret key, or null if no key was found.
     */
    public void setKey(SecretKey key) {
        this.theSecretKey = key;
    }

    /**
     * Used to obtain the secret key set within the Callback.
     *
     * @return The secret key, or null if no key was found.
     */
    public SecretKey getKey() {
        return this.theSecretKey;
    }
}
