/**
 * EasyBeans
 * Copyright (C) 2010 Bull S.A.S.
 * Contact: easybeans@objectweb.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: ClientAuthConfig.java 6201 2012-03-21 10:28:10Z benoitf $
 * --------------------------------------------------------------------------
 */
package javax.security.auth.message.config;

import java.util.Map;
import javax.security.auth.Subject;
import javax.security.auth.message.AuthException;

/**
 * This interface encapsulates the configuration of ClientAuthContext objects
 * for a message layer and application context (for example, the messaging context of
 * a specific application, or set of applications).
 *
 * <p />
 *
 * Implementations of this interface are returned by an AuthConfigProvider.
 *
 * <p />
 *
 * Callers interact with a ClientAuthConfig to obtain ClientAuthContext
 * objects suitable for processing a given message exchange at the layer and
 * within the application context of the ClientAuthConfig.
 *
 * Each ClientAuthContext object is responsible for instantiating,
 * initializing, and invoking the one or more ClientAuthModules
 * encapsulated in the ClientAuthContext.
 *
 * <p />
 * After having acquired a ClientAuthContext, a caller operates on the
 * context to cause it to invoke the encapsulated ClientAuthModules to
 * secure client requests and to validate server responses.
 *
 * @version 1.0
 * @see AuthConfigProvider
 */
public interface ClientAuthConfig extends AuthConfig {

    /**
     * Get a ClientAuthContext instance from this ClientAuthConfig.
     *
     * <p />
     *
     * The implementation of this method returns a ClientAuthContext
     * instance that encapsulates the ClientAuthModules used to
     * secure and validate requests/responses associated
     * with the given <i>authContextID</i>.
     *
     * <p />
     *
     * Specifically, this method accesses this ClientAuthConfig
     * object with the argument <i>authContextID</i> to determine the
     * ClientAuthModules that are to be encapsulated in the returned
     * ClientAuthContext instance.
     *
     * <p />
     *
     * The ClientAuthConfig object establishes the request
     * and response MessagePolicy objects that are passed to the encapsulated
     * modules when they are initialized by the returned ClientAuthContext
     * instance. It is the modules' responsibility to enforce these policies
     * when invoked.
     *
     * @param authContextID An String identifier used to index
     *		the provided <i>config</i>, or null.
     *		This value must be identical to the value returned by
     *		the <code>getAuthContextID</code> method for all
     *		<code>MessageInfo</code> objects passed to the
     *		<code>secureRequest</code>
     *		method of the returned ClientAuthContext.
     *
     * @param clientSubject A Subject that represents the source of the
     *          service request to be secured by the acquired authentication
     *          context. The principals and credentials of
     *          the Subject may be used to select or acquire the
     *          authentication context. If the Subject is not null,
     *          additional Principals or credentials (pertaining to the source
     *          of the request) may be added to the Subject. A null value may
     *          be passed for this parameter.
     *
     * @param properties A Map object that may be used by
     *          the caller to augment the properties that will be passed
     *          to the encapsulated modules at module initialization.
     *          The null value may be passed for this parameter.
     *
     * @return A ClientAuthContext instance that encapsulates the
     *		ClientAuthModules used to secure and validate
     *		requests/responses associated with the given
     *          <i>authContextID</i>,
     *		or null (indicating that no modules are configured).
     *
     * @exception AuthException If this method fails.
     */
    public abstract ClientAuthContext
            getAuthContext(String authContextID, Subject clientSubject,
            Map properties) throws AuthException;

}
