/**
 * EasyBeans
 * Copyright (C) 2010 Bull S.A.S.
 * Contact: easybeans@objectweb.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: AuthConfigFactoryTest.java 6201 2012-03-21 10:28:10Z benoitf $
 * --------------------------------------------------------------------------
 */
package javax.security.auth.message.config;

import java.security.Security;
import javax.security.auth.message.AuthException;
import org.junit.After;
import org.junit.AfterClass;
import org.junit.Assert;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;


public class AuthConfigFactoryTest {

    public AuthConfigFactoryTest() {
    }

    @BeforeClass
    public static void setUpClass() throws Exception {
    }

    @AfterClass
    public static void tearDownClass() throws Exception {
    }

    @Before
    public void setUp() {
    }

    @After
    public void tearDown() {
    }

    /**
     * Test of getFactory method, of class AuthConfigFactory.
     */
    @Test
    public void testGetFactory() throws Exception {
        System.out.println("getFactory");
        AuthConfigFactory result;

        // Security property is not set : return null
        try {
            AuthConfigFactory expected = null;
            result = AuthConfigFactory.getFactory();
            Assert.assertEquals(expected, result);
            // OK
        } catch (SecurityException se) {
            Assert.fail("getFactory with no security property should NOT"
                    + " throw an exception, but return null (Exception"
                    + " thrown: " + se.getCause() + ")");
        }

        // Set the security property to a wrong value and try again: AuthException
        Security.setProperty(
                AuthConfigFactory.DEFAULT_FACTORY_SECURITY_PROPERTY,
                "a.non.existing.factory.ImplementationClass");
        try {
            result = AuthConfigFactory.getFactory();
            // Shouldn't happen
            Assert.fail("getFactory with wrong security property should throw"
                    + " an AuthException");
        } catch (SecurityException se) {
            Assert.assertTrue(se.getCause() instanceof AuthException);
            // OK
        }

        // Set the security property to a good value but wrong factory (ABSTRACT
        // and NO zero-arg constructor) and try again: AuthException
        Security.setProperty(
                AuthConfigFactory.DEFAULT_FACTORY_SECURITY_PROPERTY,
                "javax.security.auth.message.config.AuthConfigFactoryAbsNoConstImpl");
        try {
            result = AuthConfigFactory.getFactory();
            // Shouldn't happen
            Assert.fail("getFactory with good security property but wrong"
                    + " factory class should throw an AuthException");
        } catch (SecurityException se) {
            // Verify exception message
            Assert.assertTrue(se.getCause() instanceof AuthException);
            Assert.assertEquals(
                    AuthConfigFactory.class.getName() + ": Could not instantiate"
                    + " the AuthConfigFactory instance with zero-argument"
                    + " constructor of class 'javax.security.auth.message"
                    + ".config.AuthConfigFactoryAbsNoConstImpl'"
                    ,
                    se.getCause().getMessage());
            // OK
        }

        // Set the security property to a good value but wrong factory (ABSTRACT
        // with zero-arg constructor) and try again: AuthException
        Security.setProperty(
                AuthConfigFactory.DEFAULT_FACTORY_SECURITY_PROPERTY,
                "javax.security.auth.message.config.AuthConfigFactoryAbstractImpl");
        try {
            result = AuthConfigFactory.getFactory();
            // Shouldn't happen
            Assert.fail("getFactory with good security property but wrong"
                    + " factory class should throw an AuthException");
        } catch (SecurityException se) {
            // Verify exception message
            Assert.assertTrue(se.getCause() instanceof AuthException);
            Assert.assertEquals(
                    AuthConfigFactory.class.getName() + ": Could not instantiate"
                    + " the AuthConfigFactory instance with zero-argument"
                    + " constructor of class 'javax.security.auth.message"
                    + ".config.AuthConfigFactoryAbstractImpl'"
                    ,
                    se.getCause().getMessage());
            // OK
        }

        // Set the security property to a good value but wrong factory (NOT
        // abstract but NO zero-arg constructor) and try again: AuthException
        Security.setProperty(
                AuthConfigFactory.DEFAULT_FACTORY_SECURITY_PROPERTY,
                "javax.security.auth.message.config.AuthConfigFactoryNoConstructorImpl");
        try {
            result = AuthConfigFactory.getFactory();
            // Shouldn't happen
            Assert.fail("getFactory with good security property but wrong"
                    + " factory class should throw an AuthException");
        } catch (SecurityException se) {
            // Verify exception message
            Assert.assertTrue(se.getCause() instanceof AuthException);
            Assert.assertEquals(
                    AuthConfigFactory.class.getName() + ": Could not instantiate"
                    + " the AuthConfigFactory instance with zero-argument"
                    + " constructor of class 'javax.security.auth.message"
                    + ".config.AuthConfigFactoryNoConstructorImpl'"
                    ,
                    se.getCause().getMessage());
            // OK
        }

        // Set the security property to a good value but wrong factory (NOT
        // abstract but PRIVATE zero-arg constructor) and try
        // again: AuthException
        Security.setProperty(
                AuthConfigFactory.DEFAULT_FACTORY_SECURITY_PROPERTY,
                "javax.security.auth.message.config.AuthConfigFactoryBadConstructorImpl");
        try {
            result = AuthConfigFactory.getFactory();
            // Shouldn't happen
            Assert.fail("getFactory with good security property but wrong"
                    + " factory class should throw an AuthException");
        } catch (SecurityException se) {
            // Verify exception message
            Assert.assertTrue(se.getCause() instanceof AuthException);
            Assert.assertEquals(
                    AuthConfigFactory.class.getName() + ": Could not access the"
                    + " AuthConfigFactory class 'javax.security.auth.message"
                    + ".config.AuthConfigFactoryBadConstructorImpl' or its"
                    + " zero-argument constructor"
                    ,
                    se.getCause().getMessage());
            // OK
        }

        // Set the security property to a GOOD value and a GOOD factory (NOT
        // abstract and zero-arg constructor) and try again: OK
        Security.setProperty(
                AuthConfigFactory.DEFAULT_FACTORY_SECURITY_PROPERTY,
                "javax.security.auth.message.config.AuthConfigFactoryWorkingImpl");
        try {
            result = AuthConfigFactory.getFactory();
            // OK
        } catch (SecurityException se) {
            // Shouldn't happen
            Assert.fail("getFactory with the right security property should NOT"
                    + " throw an AuthException (Exception thrown: "
                    + se.getCause().getMessage() + ")");
        }

        // Clean the security property
        Security.setProperty(AuthConfigFactory.DEFAULT_FACTORY_SECURITY_PROPERTY, "");
    }

    /**
     * Test of setFactory method, of class AuthConfigFactory.
     */
    @Test
    public void testSetFactory() {
        System.out.println("setFactory");

        AuthConfigFactory factory = new AuthConfigFactoryWorkingImpl();
        try {
            AuthConfigFactory.setFactory(factory);
            Assert.assertEquals(factory, AuthConfigFactory.getFactory());
            // OK
        } catch (SecurityException se) {
            // Shoudn't happen
            Assert.fail("getFactory when the factory already exists should NOT"
                    + " throw an AuthException (Exception thrown: "
                    + se.getCause().getMessage() + ")");
        }
    }

}
