/**
 * OW2 Specifications
 * Copyright (C) 2010 Bull S.A.S.
 * Contact: easybeans@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: EntityManagerFactory.java 5417 2010-03-24 09:04:14Z benoitf $
 * --------------------------------------------------------------------------
 */

package javax.persistence;

import java.util.Map;

import javax.persistence.criteria.CriteriaBuilder;
import javax.persistence.metamodel.Metamodel;

/**
 * Interface used to interact with the entity manager factory for the persistence unit.
 * @see <a href="http://jcp.org/en/jsr/detail?id=317">JPA 2.0 specification</a>
 * @author Florent Benoit
 * @since JPA 2.0 version.
 */
public interface EntityManagerFactory {

    /**
     * Create a new application-managed EntityManager. This method returns a new EntityManager instance each time it is invoked.
     * The isOpen method will return true on the returned instance.
     * @return entity manager instance
     * @throws IllegalStateException if the entity manager factory has been closed
     */
    public EntityManager createEntityManager();

    /**
     * Create a new application-managed EntityManager with the specified Map of properties. This method returns a new
     * EntityManager instance each time it is invoked. The isOpen method will return true on the returned instance.
     * @param map properties for entity manager
     * @return entity manager instance
     * @throws IllegalStateException if the entity manager factory has been closed
     */
    public EntityManager createEntityManager(Map map);

    /**
     * Return an instance of CriteriaBuilder for the creation of CriteriaQuery objects.
     * @return CriteriaBuilder instance
     * @throws IllegalStateException if the entity manager factory has been closed
     */
    public CriteriaBuilder getCriteriaBuilder();

    /**
     * Return an instance of Metamodel interface for access to the metamodel of the persistence unit.
     * @return Metamodel instance
     * @throws IllegalStateException if the entity manager factory has been closed
     */
    public Metamodel getMetamodel();

    /**
     * Indicates whether the factory is open. Returns true until the factory has been closed.
     * @return boolean indicating whether the factory is open
     */
    public boolean isOpen();

    /**
     * Close the factory, releasing any resources that it holds. After a factory instance has been closed, all methods invoked on
     * it will throw the IllegalStateException, except for isOpen, which will return false. Once an EntityManagerFactory has been
     * closed, all its entity managers are considered to be in the closed state.
     * @throws IllegalStateException if the entity manager factory has been closed
     */
    public void close();

    /**
     * Get the properties and associated values that are in effect for the entity manager factory. Changing the contents of the
     * map does not change the configuration in effect.
     * @return properties
     * @throws IllegalStateException if the entity manager factory has been closed
     */
    public Map<String, Object> getProperties();

    /**
     * Access the cache that is associated with the entity manager factory (the "second level cache").
     * @return instance of the Cache interface
     * @throws IllegalStateException if the entity manager factory has been closed
     */
    public Cache getCache();

    /**
     * Return interface providing access to utility methods for the persistence unit.
     * @return PersistenceUnitUtil interface
     * @throws IllegalStateException if the entity manager factory has been closed
     */
    public PersistenceUnitUtil getPersistenceUnitUtil();
}
