/**
 * OW2 Specifications
 * Copyright (C) 2010 Bull S.A.S.
 * Contact: easybeans@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: MapKeyColumn.java 5417 2010-03-24 09:04:14Z benoitf $
 * --------------------------------------------------------------------------
 */

package javax.persistence;

import static java.lang.annotation.ElementType.FIELD;
import static java.lang.annotation.ElementType.METHOD;
import static java.lang.annotation.RetentionPolicy.RUNTIME;

import java.lang.annotation.Retention;
import java.lang.annotation.Target;

/**
 * The MapKeyColumn annotation is used to specify the mapping for the key column of a map whose map key is a basic type. If the
 * name element is not specified, it defaults to the concatenation of the fol- lowing: the name of the referencing relationship
 * field or property; "_"; "KEY".
 * @see <a href="http://jcp.org/en/jsr/detail?id=317">JPA 2.0 specification</a>
 * @author Florent Benoit
 * @since JPA 2.0 version.
 */
@Target({ METHOD, FIELD })
@Retention(RUNTIME)
public @interface MapKeyColumn {
    /**
     * (Optional) The name of the map key col- umn. The table in which it is found depends upon the context. If the map key is for
     * an element collection, the map key column is in the collection table for the map value. If the map key is for a ManyToMany
     * entity relationship or for a OneToMany entity relationship using a join table, the map key column is in a join table. If
     * the map key is for a OneTo- Many entity relationship using a foreign key mapping strategy, the map key col- umn is in the
     * table of the entity that is the value of the map.
     */
    String name() default "";

    /**
     * (Optional) Whether the column is a unique key. This is a shortcut for the UniqueConstraint annotation at the table level
     * and is useful for when the unique key constraint corresponds to only a sin- gle column. This constraint applies in addition
     * to any constraint entailed by primary key mapping and to constraints specified at the table level.
     * **/
    boolean unique() default false;

    /**
     * (Optional) Whether the database column is nullable.
     */
    boolean nullable() default false;

    /**
     * (Optional) Whether the column is included in SQL INSERT statements generated by the persistence provider.
     */
    boolean insertable() default true;

    /**
     * (Optional) Whether the column is included in SQL UPDATE statements generated by the persistence provider.
     */
    boolean updatable() default true;

    /**
     * (Optional) The SQL fragment that is used when generating the DDL for the column.
     */
    String columnDefinition() default "";

    /**
     * (Optional) The name of the table that contains the column.
     **/
    String table() default "";

    /**
     * (Optional) The column length. (Applies only if a string-valued column is used.)
     **/
    int length() default 255;

    /**
     * (Optional) The precision for a decimal (exact numeric) column. (Applies only if a decimal column is used.)
     **/
    int precision() default 0;

    /**
     * (Optional) The scale for a decimal (exact numeric) column. (Applies only if a dec- imal column is used.)
     **/
    int scale() default 0;
}
