/**
 * OW2 Specifications
 * Copyright (C) 2007 Bull S.A.S.
 * Contact: easybeans@objectweb.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: ConnectionFactory.java 6201 2012-03-21 10:28:10Z benoitf $
 * --------------------------------------------------------------------------
 */

package javax.resource.cci;

import java.io.Serializable;

import javax.resource.Referenceable;
import javax.resource.ResourceException;

/**
 * ConnectionFactory provides an interface for getting connection to
 * an EIS instance. An implementation of ConnectionFactory interface
 * is provided by a resource adapter.<br/>
 *
 * Application code looks up a ConnectionFactory instance from JNDI
 * namespace and uses it to get EIS connections.<br/>
 *
 * An implementation class for ConnectionFactory is required to
 * implement java.io.Serializable and javax.resource.Referenceableinterfaces
 * to support JNDI registration.
 * @author Guillaume Sauthier
 */
public interface ConnectionFactory extends Serializable, Referenceable {

    /**
     * Gets a connection to an EIS instance. This getConnection variant
     * should be used when a component wants the container to manage EIS
     * sign-on. This case is termed container-managed sign-on. The
     * component does not pass any security information.
     * @return Connection instance
     * @throws ResourceException Failed to get a connection to the EIS
     *         instance. Examples of error cases are:<ul>
     *         <li>Invalid configuration of ManagedConnectionFactory--
     *         example: invalid server name</li>
     *         <li>Application server-internal error--example: connection
     *         pool related error</li>
     *         <li>Communication error</li>
     *         <li>EIS-specific error--example: EIS not active</li>
     *         <li>Resource adapter-internal error</li>
     *         <li>Security related error; example: invalid user</li>
     *         <li>Failure to allocate system resources</li>
     *         </ul>
     */
    Connection getConnection() throws ResourceException;

    /**
     * Gets a connection to an EIS instance. A component should use the
     * getConnection variant with javax.resource.cci.ConnectionSpec parameter,
     * if it needs to pass any resource adapter specific security information
     * and connection parameters. In the component- managed sign-on case, an
     * application component passes security information (example: username,
     * password) through the ConnectionSpec instance.<br/>
     *
     * It is important to note that the properties passed through the
     * getConnection method should be client-specific (example: username, password,
     * language) and not related to the configuration of a target EIS instance
     * (example: port number, server name). The ManagedConnectionFactory instance
     * is configured with complete set of properties required for the creation
     * of a connection to an EIS instance.
     * @param properties Connection parameters and security information specified
     *        as ConnectionSpec instance
     * @return Connection instance
     * @throws ResourceException Failed to get a connection to the EIS instance.
     *         Examples of error cases are:<ul>
     *         <li>Invalid specification of input parameters</li>
     *         <li>Invalid configuration of ManagedConnectionFactory-- example:
     *         invalid server name</li>
     *         <li>Application server-internal error--example: connection pool
     *         related error</li>
     *         <li>Communication error</li>
     *         <li>EIS-specific error--example: EIS not active</li>
     *         <li>Resource adapter-internal error</li>
     *         <li>Security related error; example: invalid user</li>
     *         <li>Failure to allocate system resources</li>
     *         </ul>
     */
    Connection getConnection(ConnectionSpec properties) throws ResourceException;

    /**
     * Gets a RecordFactory instance. The RecordFactory is used for the creation
     * of generic Record instances.
     * @return RecordFactory RecordFactory instance
     * @throws ResourceException Failed to create a RecordFactory
     * @throws NotSupportedException Operation not supported
     */
    RecordFactory getRecordFactory() throws ResourceException;

    /**
     * Gets metadata for the Resource Adapter. Note that the metadata information
     * is about the ResourceAdapter and not the EIS instance. An invocation of
     * this method does not require that an active connection to an EIS instance
     * should have been established.
     * @return ResourceAdapterMetaData instance
     * @throws ResourceException Failed to get metadata information about the
     *         resource adapter
     */
    ResourceAdapterMetaData getMetaData() throws ResourceException;
}
