/**
 * EasyBeans
 * Copyright (C) 2009-2011 Bull S.A.S.
 * Contact: easybeans@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: Timer.java 6008 2011-10-17 13:41:22Z benoitf $
 * --------------------------------------------------------------------------
 */

package javax.ejb;

import java.io.Serializable;
import java.util.Date;

/**
 * The Timer interface contains information about a timer that was created
 * through the EJB Timer Service.
 * @see <a href="http://www.jcp.org/en/jsr/detail?id=318">EJB 3.1 specification</a>
 * @author Florent Benoit
 */
public interface Timer {

    /**
     * Cause the timer and all its associated expiration notifications to be
     * cancelled.
     * @throws IllegalStateException If this method is invoked while the
     *         instance is in a state that does not allow access to this method.
     * @throws NoSuchObjectLocalException If invoked on a timer that has expired
     *         or has been cancelled.
     * @throws EJBException If this method could not complete due to a
     *         system-level failure.
     */
    void cancel() throws IllegalStateException, NoSuchObjectLocalException, EJBException;

    /**
     * Get the number of milliseconds that will elapse before the next scheduled
     * timer expiration.
     * @return the number of milliseconds that will elapse before the next
     *         scheduled timer expiration.
     * @throws IllegalStateException If this method is invoked while the
     *         instance is in a state that does not allow access to this method.
     * @throws NoSuchObjectLocalException If invoked on a timer that has expired
     *         or has been cancelled.
     * @throws EJBException If this method could not complete due to a
     *         system-level failure.
     * @throws NoMoreTimeoutsException (Since EJB 3.1) Indicates that the 
     * timer has no future timeouts
     */
    long getTimeRemaining() throws IllegalStateException, NoSuchObjectLocalException, NoMoreTimeoutsException, EJBException;

    /**
     * Get the point in time at which the next timer expiration is scheduled to
     * occur.
     * @return the point in time at which the next timer expiration is scheduled
     *         to occur.
     * @throws IllegalStateException If this method is invoked while the
     *         instance is in a state that does not allow access to this method.
     * @throws NoSuchObjectLocalException If invoked on a timer that has expired
     *         or has been cancelled.
     * @throws EJBException If this method could not complete due to a
     *         system-level failure.
     * @throws NoMoreTimeoutsException (Since EJB 3.1) Indicates that the 
     * timer has no future timeouts
     */
    Date getNextTimeout() throws IllegalStateException, NoSuchObjectLocalException, NoMoreTimeoutsException, EJBException;

    /**
     * Get the information associated with the timer at the time of creation.
     * @return The Serializable object that was passed in at timer creation, or
     *         null if the info argument passed in at timer creation was null.
     * @throws IllegalStateException If this method is invoked while the
     *         instance is in a state that does not allow access to this method.
     * @throws NoSuchObjectLocalException If invoked on a timer that has expired
     *         or has been cancelled.
     * @throws EJBException If this method could not complete due to a
     *         system-level failure.
     */
    Serializable getInfo() throws IllegalStateException, NoSuchObjectLocalException, EJBException;

    /**
     * Get a serializable handle to the timer. This handle can be used at a
     * later time to re-obtain the timer reference.
     * @return a serializable handle to the timer.
     * @throws IllegalStateException If this method is invoked while the
     *         instance is in a state that does not allow access to this method.
     * @throws NoSuchObjectLocalException If invoked on a timer that has expired
     *         or has been cancelled.
     * @throws EJBException If this method could not complete due to a
     *         system-level failure.
     */
    TimerHandle getHandle() throws IllegalStateException, NoSuchObjectLocalException, EJBException;

    /**
     * Get the schedule expression corresponding to this timer.  The timer
     * must be a calendar-based timer.  It may have been created automatically
     * or programmatically.
     * @return schedule expression for the timer.
     * @throws IllegalStateException If this method is
     * invoked while the instance is in a state that does not allow access 
     * to this method.  Also thrown if invoked on a timer that is not a
     * calendar-based timer.
     * @throws NoSuchObjectLocalException If invoked on a timer
     * that has expired or has been cancelled.
     * @throws EJBException If this method could not complete due
     * to a system-level failure.
     * @since EJB 3.1 version.
     */
    ScheduleExpression getSchedule() throws IllegalStateException, NoSuchObjectLocalException, EJBException;
    
    /**
     * Return whether this timer is a calendar-based timer.
     * @return boolean indicating whether the timer is calendar-based.
     * @throws java.lang.IllegalStateException If this method is invoked while the instance is in a state that does not allow
     * access to this method.
     * @throws javax.ejb.NoSuchObjectLocalException If invoked on a timer that has expired or has been cancelled.
     * @throws javax.ejb.EJBException If this method could not complete due to a system-level failure.
     * @since EJB 3.1 version.
     */
    boolean isCalendarTimer() throws IllegalStateException, NoSuchObjectLocalException, EJBException;  
    
    /**
     * Return whether this timer has persistent semantics.
     * @return boolean indicating whether the timer is persistent.
     * @throws IllegalStateException If this method is invoked while the instance is in a state that does not allow access to this
     * method.
     * @throws NoSuchObjectLocalException If invoked on a timer that has expired or has been cancelled.
     * @throws EJBException If this method could not complete due to a system-level failure.
     * @since EJB 3.1 version.
     */
    boolean isPersistent() throws IllegalStateException, NoSuchObjectLocalException, EJBException;

}
