/**
 * OW2 Specifications
 * Copyright (C) 2011 Bull S.A.S.
 * Contact: easybeans@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 *
 * --------------------------------------------------------------------------
 * $Id: ForwardingRuntimeDelegateTestCase.java 5871 2011-05-05 12:11:35Z benoitf $
 * --------------------------------------------------------------------------
 */

package org.ow2.spec.ee.jaxrs;

import javax.ws.rs.ext.RuntimeDelegate;

import org.osgi.framework.BundleContext;
import org.osgi.framework.ServiceReference;
import org.testng.Assert;
import org.testng.annotations.Test;

import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;
import static org.testng.Assert.fail;

public class ForwardingRuntimeDelegateTestCase {

    @Test(expectedExceptions = IllegalStateException.class)
    public void testIllegalStateExceptionIsThrownForCreateEndpointWhenNoProviderIsRegistered() throws Exception {
        BundleContext context = mock(BundleContext.class);
        ForwardingRuntimeDelegate delegate = new ForwardingRuntimeDelegate(context);
        delegate.start();
        delegate.createEndpoint(null, null);
    }

    @Test(expectedExceptions = IllegalStateException.class)
    public void testIllegalStateExceptionIsThrownForCreateHeaderDelegateWhenNoProviderIsRegistered() throws Exception {
        BundleContext context = mock(BundleContext.class);
        ForwardingRuntimeDelegate delegate = new ForwardingRuntimeDelegate(context);
        delegate.start();
        delegate.createHeaderDelegate(null);
    }

    @Test(expectedExceptions = IllegalStateException.class)
    public void testIllegalStateExceptionIsThrownForCreateResponseBuilderWhenNoProviderIsRegistered() throws Exception {
        BundleContext context = mock(BundleContext.class);
        ForwardingRuntimeDelegate delegate = new ForwardingRuntimeDelegate(context);
        delegate.start();
        delegate.createResponseBuilder();
    }

    @Test(expectedExceptions = IllegalStateException.class)
    public void testIllegalStateExceptionIsThrownForCreateVariantListBuilderWhenNoProviderIsRegistered() throws Exception {
        BundleContext context = mock(BundleContext.class);
        ForwardingRuntimeDelegate delegate = new ForwardingRuntimeDelegate(context);
        delegate.start();
        delegate.createVariantListBuilder();
    }

    @Test(expectedExceptions = IllegalStateException.class)
    public void testIllegalStateExceptionIsThrownForCreateUriBuilderWhenNoProviderIsRegistered() throws Exception {
        BundleContext context = mock(BundleContext.class);
        ForwardingRuntimeDelegate delegate = new ForwardingRuntimeDelegate(context);
        delegate.start();
        delegate.createUriBuilder();
    }

    @Test
    public void testWithOneRuntimeDelegate() throws Exception {
        BundleContext context = mock(BundleContext.class);
        ServiceReference reference = mock(ServiceReference.class);
        RuntimeDelegate service = mock(RuntimeDelegate.class);

        when(context.getService(reference)).thenReturn(service);

        ForwardingRuntimeDelegate delegate = new ForwardingRuntimeDelegate(context);
        delegate.start();

        delegate.onRegistration(reference);

        delegate.createHeaderDelegate(null);
        delegate.createUriBuilder();
        delegate.createResponseBuilder();
        delegate.createVariantListBuilder();
        delegate.createEndpoint(null, null);

    }

    @Test(expectedExceptions = IllegalStateException.class)
    public void testWithOneRuntimeDelegateDisappearing() throws Exception {
        BundleContext context = mock(BundleContext.class);
        ServiceReference reference = mock(ServiceReference.class);
        RuntimeDelegate service = mock(RuntimeDelegate.class);

        when(context.getService(reference)).thenReturn(service);

        ForwardingRuntimeDelegate delegate = new ForwardingRuntimeDelegate(context);
        delegate.start();

        delegate.onRegistration(reference);

        try {
            delegate.createHeaderDelegate(null);
            delegate.createUriBuilder();
            delegate.createResponseBuilder();
            delegate.createVariantListBuilder();
            delegate.createEndpoint(null, null);
        } catch (Throwable t) {
            fail("No Exception expected at this point (1 service available)", t);
        }

        // it disappear
        delegate.onUnregistration(reference);

        // should throw an Exception
        delegate.createUriBuilder();

    }

    @Test(expectedExceptions = IllegalStateException.class)
    public void testWithTwoRuntimeDelegateAtStartupThenTheUsedOneDisappear() throws Exception {
        BundleContext context = mock(BundleContext.class);
        ServiceReference reference1 = mock(ServiceReference.class);
        ServiceReference reference2 = mock(ServiceReference.class);
        RuntimeDelegate service1 = mock(RuntimeDelegate.class);
        RuntimeDelegate service2 = mock(RuntimeDelegate.class);

        when(context.getService(reference1)).thenReturn(service1);
        when(context.getService(reference2)).thenReturn(service2);

        ForwardingRuntimeDelegate delegate = new ForwardingRuntimeDelegate(context);
        delegate.start();

        delegate.onRegistration(reference1);
        delegate.onRegistration(reference2);

        try {
            delegate.createHeaderDelegate(null);
            delegate.createUriBuilder();
            delegate.createResponseBuilder();
            delegate.createVariantListBuilder();
            delegate.createEndpoint(null, null);
        } catch (Throwable t) {
            fail("No Exception expected at this point (multiple services available)", t);
        }

        // service1 disappear
        delegate.onUnregistration(reference1);

        // Should still works (using service 2)
        try {
            delegate.createUriBuilder();
        } catch (Throwable t) {
            fail("No Exception expected at this point (still have 1 service available)", t);
        }

        // service2 disappear
        delegate.onUnregistration(reference2);

        // this time it fails
        delegate.createUriBuilder();

    }

}
