/**
 * EasyBeans
 * Copyright (C) 2006 Bull S.A.S.
 * Contact: easybeans@objectweb.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: PersistenceContext.java 4359 2008-12-10 10:24:40Z sauthieg $
 * --------------------------------------------------------------------------
 */

package javax.persistence;

import static java.lang.annotation.ElementType.FIELD;
import static java.lang.annotation.ElementType.METHOD;
import static java.lang.annotation.ElementType.TYPE;
import static java.lang.annotation.RetentionPolicy.RUNTIME;
import static javax.persistence.PersistenceContextType.TRANSACTION;

import java.lang.annotation.Retention;
import java.lang.annotation.Target;

/**
 * The PersistenceContext annotation is used to express a dependency on a
 * container-managed entity manager persistence context.
 * @see <a href="http://www.jcp.org/en/jsr/detail?id=220">EJB 3.0 specification</a>
 * @author Florent Benoit
 * @since EJB 3.0 version.
 */
@Target({ TYPE, METHOD, FIELD })
@Retention(RUNTIME)
public @interface PersistenceContext {

    /**
     * The name element refers to the name by which the entity manager is to be
     * accessed in the environment referencing context, and is not needed when
     * dependency injection is used.
     */
    String name() default "";

    /**
     * The optional unitName element refers to the name of the persistence unit.
     * If the unitName element is specified, the persistence unit for the entity
     * manager that is accessible in JNDI must have the same name.
     */
    String unitName() default "";

    /**
     * The type element specifies whether a transaction-scoped or extended
     * persistence context is to be used. If the type element is not specified,
     * a transaction-scoped persistence context is used.
     */
    PersistenceContextType type() default TRANSACTION;

    /**
     * The optional properties element may be used to specify properties for the
     * container or persistence provider. Vendor specific properties may be
     * included in the set of properties, and are passed to the persistence
     * provider by the container when the entity manager is created. Properties
     * that are not recognized by a vendor must be ignored.
     */
    PersistenceProperty[] properties() default {};
}
