/**
 * OW2 Specifications
 * Copyright (C) 2010 Bull S.A.S.
 * Contact: easybeans@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: FetchParent.java 5417 2010-03-24 09:04:14Z benoitf $
 * --------------------------------------------------------------------------
 */

package javax.persistence.criteria;

import javax.persistence.metamodel.PluralAttribute;
import javax.persistence.metamodel.SingularAttribute;

/**
 * Represents an element of the from clause which may
 * function as the parent of Fetches.
 * @param <Z> the source type
 * @param <X> the target type
 * @see <a href="http://jcp.org/en/jsr/detail?id=317">JPA 2.0 specification</a>
 * @author Florent Benoit
 * @since JPA 2.0 version.
 */
public interface FetchParent<Z, X> {
    /**
     * Return the fetch joins that have been made from this type.
     * Returns empty set if no fetch joins have been made from
     * this type.
     * Modifications to the set do not affect the query.
     * @return fetch joins made from this type
     */
    java.util.Set<Fetch<X, ?>> getFetches();
    /**
     *  Create a fetch join to the specified single-valued attribute
     *  using an inner join.
     *  @param attribute target of the join
     *  @return the resulting fetch join
     */
    <Y> Fetch<X, Y> fetch(SingularAttribute<? super X, Y> attribute);
    /**
     *  Create a fetch join to the specified single-valued attribute
     *  using the given join type.
     *  @param attribute target of the join
     *  @param jt join type
     *  @return the resulting fetch join
     */
    <Y> Fetch<X, Y> fetch(SingularAttribute<? super X, Y> attribute,
            JoinType jt);
    /**
     *  Create a fetch join to the specified collection-valued
     *  attribute using an inner join.
     *  @param attribute target of the join
     *  @return the resulting join
     */
    <Y> Fetch<X, Y> fetch(PluralAttribute<? super X, ?, Y> attribute);
    /**
     *  Create a fetch join to the specified collection-valued
     *  attribute using the given join type.
     *  @param attribute target of the join
     *  @param jt join type
     *  @return the resulting join
     */
    <Y> Fetch<X, Y> fetch(PluralAttribute<? super X, ?, Y> attribute,
            JoinType jt);
    //String-based:
    /**
     *  Create a fetch join to the specified attribute using an
     *  inner join.
     *  @param attributeName name of the attribute for the
     *         target of the join
     *  @return the resulting fetch join
     *  @throws IllegalArgumentException if attribute of the given
     *          name does not exist
     */
    <X, Y> Fetch<X, Y> fetch(String attributeName);
    /**
     *  Create a fetch join to the specified attribute using
     *  the given join type.
     *  @param attributeName name of the attribute for the
     *         target of the join
     *  @param jt join type
     *  @return the resulting fetch join
     *  @throws IllegalArgumentException if attribute of the given
     *          name does not exist
     */
    <X, Y> Fetch<X, Y> fetch(String attributeName, JoinType jt);

}
