/**
 * OW2 Specifications
 * Copyright (C) 2007 Bull S.A.S.
 * Contact: easybeans@objectweb.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 *
 * --------------------------------------------------------------------------
 * $Id: WebParam.java 4359 2008-12-10 10:24:40Z sauthieg $
 * --------------------------------------------------------------------------
 */

package javax.jws;

import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;

/**
 * Customizes the mapping of an individual parameter to a Web Service
 * message part and XML element.
 * @author Guillaume Sauthier
 */
@Retention(value=RetentionPolicy.RUNTIME)
@Target(ElementType.PARAMETER)
public @interface WebParam {

    /**
     * The direction in which the parameter flows.
     */
    public static enum Mode {
        IN,
        INOUT,
        OUT
    }

    /**
     * If true, the parameter is pulled from a message header
     * rather then the message body.
     */
    boolean header() default false;

    /**
     * The direction in which the parameter is flowing (One of IN,
     * OUT, or INOUT).<br/>
     *
     * The OUT and INOUT modes may only be specified for parameter types
     * that conform to the definition of Holder types (JAX-WS 2.0 [5],
     * section 2.3.3). Parameters that are Holder Types MUST be OUT or
     * INOUT.
     */
    Mode mode() default Mode.IN;

    /**
     * Name of the parameter.<br/>
     *
     * If the operation is rpc style and @WebParam.partName has not been
     * specified, this is name of the wsdl:part representing the parameter.<br/>
     *
     * If the operation is document style or the parameter maps to a header,
     * this is the local name of the XML element representing the parameter.<br/>
     *
     * A name MUST be specified if the operation is document style, the
     * parameter style is BARE, and the mode is OUT or INOUT.
     */
    String name() default "";

    /**
     * The name of the wsdl:part representing this parameter.<br/>
     *
     * This is only used if the operation is rpc style or if the operation
     * is document style and the parameter style is BARE.
     *
     * @since 2.0
     */
    String partName() default "";

    /**
     * The XML namespace for the parameter.<br/>
     *
     * Only used if the operation is document style or the parameter maps
     * to a header. If the target namespace is set to ??, this represents
     * the empty namespace.
     */
    String targetNamespace() default "";
}
