/**
 * OW2 Spec
 * Copyright (C) 2010 Bull S.A.S.
 * Contact: easybeans@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: Inject.java 5374 2010-02-25 17:22:19Z sauthieg $
 * --------------------------------------------------------------------------
 */

package javax.inject;

import java.lang.annotation.Documented;
import java.lang.annotation.Retention;
import static java.lang.annotation.RetentionPolicy.RUNTIME;
import static java.lang.annotation.ElementType.METHOD;
import static java.lang.annotation.ElementType.CONSTRUCTOR;
import static java.lang.annotation.ElementType.FIELD;
import java.lang.annotation.Target;

/**
 * Identifies injectable constructors, methods, and fields. May apply to static as well as
 * instance members. An injectable member may have any access modifier (private, package-private,
 * protected, public). Constructors are injected first, followed by fields, and then methods.
 * Fields and methods in superclasses are injected before those in subclasses. Ordering of injection
 * among fields and among methods in the same class is not specified.
 *
 * Injectable constructors are annotated with <code>@Inject</code> and accept zero or more dependencies
 * as arguments. <code>@Inject</code> can apply to at most one constructor per class.
 *
 * <pre>
 * &#064;Inject ConstructorModifiers<sub>opt</sub> SimpleTypeName(FormalParameterListopt) Throws<sub>opt</sub> ConstructorBody
 * </pre>
 *
 * &#064;Inject is optional for public, no-argument constructors when no other constructors are present.
 * This enables injectors to invoke default constructors.
 *
 * <pre>
 *  &#064;Inject<sub>opt</sub> Annotations<sub>opt</sub> public SimpleTypeName() Throws<sub>opt</sub> ConstructorBody
 * </pre>
 *
 * Injectable fields:
 * <ul>
 *  <li>are annotated with @Inject.</li>
 *  <li>are not final.</li>
 *  <li>may have any otherwise valid name.</li>
 * </ul>
 *
 * <pre>
 *  &#064;Inject FieldModifiers<sub>opt</sub> Type VariableDeclarators;
 * </pre>
 *
 * Injectable methods:
 * <ul>
 *  <li>are annotated with @Inject.</li>
 *  <li>are not abstract.</li>
 *  <li>do not declare type parameters of their own.</li>
 *  <li>may return a result</li>
 *  <li>may have any otherwise valid name.</li>
 *  <li>accept zero or more dependencies as arguments.</li>
 * </ul>
 *
 * <pre>
 *  &#064;Inject MethodModifiers<sub>opt</sub> ResultType Identifier(FormalParameterList<sub>opt</sub>) Throws<sub>opt</sub> MethodBody
 * </pre>
 *
 * The injector ignores the result of an injected method, but non-void return types are allowed to
 * support use of the method in other contexts (builder-style method chaining, for example).
 *
 * Examples:
 * <pre>
 *  public class Car {
 *    // Injectable constructor
 *    &#064;Inject public Car(Engine engine) { ... }
 *    // Injectable field
 *    &#064;Inject private Provider&lt;Seat> seatProvider;
 *    // Injectable package-private method
 *    &#064;Inject void install(Windshield windshield, Trunk trunk) { ... }
 *  }
 * </pre>
 *
 * A method annotated with <code>@Inject</code> that overrides another method annotated with
 * <code>@Inject</code> will only be injected once per injection request per instance. A method with
 * no <code>@Inject</code> annotation that overrides a method annotated with <code>@Inject</code> will not be injected.
 *
 * Injection of members annotated with <code>@Inject</code> is required. While an injectable member
 * may use any accessibility modifier (including private), platform or injector limitations (like
 * security restrictions or lack of reflection support) might preclude injection of non-public members.
 *
 * <h1>Qualifiers</h1>
 * A qualifier may annotate an injectable field or parameter and, combined with the type, identify the
 * implementation to inject. Qualifiers are optional, and when used with @Inject in injector-independent
 * classes, no more than one qualifier should annotate a single field or parameter. The qualifiers are bold
 * in the following example:
 *
 * <pre>
 *  public class Car {
 *    &#064;Inject private @Leather Provider&lt;Seat> seatProvider;
 *
 *    &#064;Inject void install(&#064;Tinted Windshield windshield,
 *                         &#064;Big Trunk trunk) { ... }
 *  }
 * </pre>
 *
 * If one injectable method overrides another, the overriding method's parameters do not automatically
 * inherit qualifiers from the overridden method's parameters.
 *
 * <h1>Injectable Values</h1>
 *
 * For a given type T and optional qualifier, an injector must be able to inject a user-specified class that:
 * <ol>
 *  <li>is assignment compatible with T and</li>
 *  <li>has an injectable constructor.</li>
 * </ol>
 *
 * For example, the user might use external configuration to pick an implementation of T. Beyond that, which
 * values are injected depend upon the injector implementation and its configuration.
 *
 * <h1>Circular Dependencies</h1>
 *
 * Detecting and resolving circular dependencies is left as an exercise for the injector implementation.
 * Circular dependencies between two constructors is an obvious problem, but you can also have a circular
 * dependency between injectable fields or methods:
 *
 * <pre>
 *  class A {
 *    &#064;Inject B b;
 *  }
 *  class B {
 *    &#064;Inject A a;
 *  }
 * </pre>
 *
 * When constructing an instance of A, a naive injector implementation might go into an infinite loop
 * constructing an instance of B to set on A, a second instance of A to set on B, a second instance
 * of B to set on the second instance of A, and so on.
 *
 * A conservative injector might detect the circular dependency at build time and generate an error, at
 * which point the programmer could break the circular dependency by injecting Provider&lt;A> or Provider&lt;B> instead
 * of A or B respectively. Calling get() on the provider directly from the constructor or method
 * it was injected into defeats the provider's ability to break up circular dependencies. In the case of
 * method or field injection, scoping one of the dependencies (using singleton scope, for example) may also
 * enable a valid circular relationship.
 *
 * @author Guillaume Sauthier
 *
 * @since DI 1.0
 *
 * @see javax.inject.Provider
 * @see javax.inject.Qualifier
 */
@Target({METHOD, CONSTRUCTOR, FIELD})
@Retention(RUNTIME)
@Documented
public @interface Inject {}
