/**
 * OW2 Spec
 * Copyright (C) 2010 Bull S.A.S.
 * Contact: easybeans@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: Context.java 5375 2010-02-25 17:25:09Z sauthieg $
 * --------------------------------------------------------------------------
 */

package javax.enterprise.context.spi;

import java.lang.annotation.Annotation;

import javax.enterprise.context.ContextNotActiveException;

/**
 * Provides an operation for obtaining contextual instances with a particular scope of any contextual
 * type. Any instance of Context is called a context object.
 *
 * The context object is responsible for creating and destroying contextual instances by calling
 * operations of {@link javax.enterprise.context.spi.Contextual}. In particular, the context object
 * is responsible for destroying any contextual instance it creates by passing the instance to
 * {@link javax.enterprise.context.spi.Contextual#destroy(Object, CreationalContext)}. A destroyed instance
 * must not subsequently be returned by {@code get()}. The context object must pass the same instance of
 * {@link javax.enterprise.context.spi.CreationalContext} to {@code Contextual.destroy()}
 * that it passed to {@code Contextual.create()} when it created the instance.
 *
 * A custom context object may be registered with the container using
 * {@link javax.enterprise.inject.spi.AfterBeanDiscovery#addContext(Context)}.
 *
 * @author Guillaume Sauthier
 */
public interface Context {

    /**
     * Get the scope type of the context object.
     *
     * @return the scope
     */
    public Class<? extends Annotation> getScope();

    /**
     * Return an existing instance of certain contextual type or create a new  instance by calling
     * {@link javax.enterprise.context.spi.Contextual#create(CreationalContext)} and return the new instance.
     *
     * @param <T> the type of contextual type
     * @param contextual the contextual type
     * @param creationalContext the context in which the new instance will be created
     * @return the contextual instance
     *
     * @throws ContextNotActiveException if the context is not active
     */
    public <T> T get(Contextual<T> contextual, CreationalContext<T> creationalContext);

    /**
     * Return an existing instance of a certain contextual type or a null value.
     *
     * @param <T> the type of the contextual type
     * @param contextual the contextual type
     * @return the contextual instance, or a null value
     *
     * @throws ContextNotActiveException if the context is not active
     */
    public <T> T get(Contextual<T> contextual);

    /**
     * Determines if the context object is active.
     *
     * @return <tt>true</tt> if the context is active, or <tt>false</tt> otherwise.
     */
    boolean isActive();
}
