/**
 * OW2 Spec
 * Copyright (C) 2010 Bull S.A.S.
 * Contact: easybeans@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: Instance.java 5375 2010-02-25 17:25:09Z sauthieg $
 * --------------------------------------------------------------------------
 */

package javax.enterprise.inject;

import java.lang.annotation.Annotation;

import javax.enterprise.util.TypeLiteral;
import javax.inject.Provider;

/**
 * Allows the application to dynamically obtain instances of beans with a specified
 * combination of required type and qualifiers.
 *
 * In certain situations, injection is not the most convenient way to obtain a contextual
 * reference. For example, it may not be used when:
 * <ul>
 *  <li>the bean type or qualifiers vary dynamically at runtime, or</li>
 *  <li>depending upon the deployment, there may be no bean which satisfies the type and qualifiers, or</li>
 *  <li>we would like to iterate over all beans of a certain type.</li>
 * </ul>
 *
 * In these situations, an instance of the Instance may be injected:
 * <pre>
 *  &#064;Inject Instance&lt;PaymentProcessor> paymentProcessor;
 * </pre>
 *
 * Any combination of qualifiers may be specified at the injection point:
 * <pre>
 *  &#064;Inject @PayBy(CHEQUE) Instance&lt;PaymentProcessor> chequePaymentProcessor;
 * </pre>
 *
 * Or, the @{@link Any} qualifier may be used, allowing the application to specify qualifiers dynamically:
 * <pre>
 *  &#064;Inject @Any Instance&lt;PaymentProcessor> anyPaymentProcessor;
 * </pre>
 *
 * Finally, the @{@link New} qualifier may be used, allowing the application to obtain a @{@link New} qualified bean:
 * <pre>
 *  &#064;Inject @New(ChequePaymentProcessor.class)
 *  Instance&lt;PaymentProcessor> chequePaymentProcessor;
 * </pre>
 *
 * For an injected Instance:
 * <ul>
 *  <li>the required type is the type parameter specified at the injection point, and</li>
 *  <li>the required qualifiers are the qualifiers specified at the injection point.</li>
 * </ul>
 *
 * The inherited {@code Provider.get()} method returns a contextual references for the unique
 * bean that matches the required type and required qualifiers and is eligible for injection
 * into the class into which the parent Instance was injected, or throws an {@link UnsatisfiedResolutionException}
 * or {@link AmbiguousResolutionException}.
 * <pre>
 *  PaymentProcessor pp = chequePaymentProcessor.get();
 * </pre>
 *                                *
 * The inherited {@code Iterable.iterator()} method returns an iterator over contextual references
 * for beans that match the required type and required qualifiers and are eligible for injection
 * into the class into which the parent Instance was injected.
 * <pre>
 *  for (PaymentProcessor pp : anyPaymentProcessor) pp.test();
 * </pre>
 *
 * @author Guillaume Sauthier
 */
public interface Instance<T> extends Iterable<T>, Provider<T> {

    /**
     * Obtains a child {@code Instance} for the given additional required qualifiers.
     * 
     * @param qualifiers the additional required qualifiers 
     * @return the child Instance
     * @throws IllegalArgumentException if passed two instances of the same qualifier type, or
     *         an instance of an annotation that is not a qualifier type
     */
    Instance<T> select(Annotation... qualifiers);

    /**
     * Obtains a child {@code Instance} for the given required type and additional required qualifiers.
     * 
     * @param subtype a {@code Class} representing the required type
     * @param qualifiers the additional required qualifiers
     * @param <U> the required type
     * @return the child Instance
     * @throws IllegalArgumentException if passed two instances of the same qualifier type, or an
     *         instance of an annotation that is not a qualifier type
     */
    <U extends T> Instance<U> select(Class<U> subtype,
                                     Annotation... qualifiers);

    /**
     * Obtains a child Instance for the given required type and additional required qualifiers.
     *
     * @param subtype a {@link TypeLiteral} representing the required type
     * @param qualifiers the additional required qualifiers 
     * @param <U> the required type
     * @return the child Instance
     * @throws IllegalArgumentException if passed two instances of the same qualifier type, or an
     *         instance of an annotation that is not a qualifier type
     */
    <U extends T> Instance<U> select(TypeLiteral<U> subtype,
                                     Annotation... qualifiers);

    /**
     * Determines if there is no bean that matches the required type and qualifiers and is eligible
     * for injection into the class into which the parent {@code Instance} was injected.
     *
     * @return <tt>true</tt> if there is no bean that matches the required type and qualifiers and is
     *         eligible for injection into the class into which the parent {@code Instance} was injected,
     *         or <tt>false</tt> otherwise.
     */
    boolean isUnsatisfied();

    /**
     * Determines if there is more than one bean that matches the required type and qualifiers and is
     * eligible for injection into the class into which the parent {@code Instance} was injected.
     *
     * @return <tt>true</tt> if there is more than one bean that matches the required type and qualifiers
     *         and is eligible for injection into the class into which the parent {@code Instance} was
     *         injected, or <tt>false</tt> otherwise.
     */
    boolean isAmbiguous();

}
