/**
 * OW2 Spec
 * Copyright (C) 2010 Bull S.A.S.
 * Contact: easybeans@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: TypeLiteral.java 5375 2010-02-25 17:25:09Z sauthieg $
 * --------------------------------------------------------------------------
 */

package javax.enterprise.util;

import java.io.Serializable;
import java.lang.reflect.GenericArrayType;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;

/**
 * Supports inline instantiation of objects that represent parameterized types with actual
 * type parameters.
 *
 * An object that represents any parameterized type may be obtained by subclassing {@code TypeLiteral}.
 * <pre>
 *  TypeLiteral&lt;List&lt;String>> stringListType = new TypeLiteral&lt;List&lt;String>>() {};
 * </pre>
 *
 * @author Guillaume Sauthier
 * @see javax.enterprise.inject.Instance#select(TypeLiteral, java.lang.annotation.Annotation...)
 * @see javax.enterprise.event.Event#select(TypeLiteral, java.lang.annotation.Annotation...) 
 */
public abstract class TypeLiteral<T> implements Serializable {

    /**
     * Store the actual type (direct subclass of TypeLiteral).
     */
    private transient Type type;

    /**
     * Store the actual raw parameter type.
     */
    private transient Class<T> rawType;

    protected TypeLiteral() {
    }

    /**
     * @return the actual type represented by this object
     */
    public final Type getType() {
        if (type == null) {
            // Get the class that directly extends TypeLiteral<?>
            Class<?> typeLiteralSubclass = getTypeLiteralSubclass(this.getClass());
            if (typeLiteralSubclass == null) {
                throw new RuntimeException(getClass() + " is not a subclass of TypeLiteral<T>");
            }

            // Get the type parameter of TypeLiteral<T> (aka the T value)
            type = getTypeParameter(typeLiteralSubclass);
            if (type == null) {
                throw new RuntimeException(getClass() + " does not specify the type parameter T of TypeLiteral<T>");
            }
        }
        return type;
    }

    /**
     * @return the raw type represented by this object
     */
    @SuppressWarnings("unchecked")
    public final Class<T> getRawType() {

        if (rawType == null) {

            // Get the actual type
            Type type = getType();
            if (type instanceof Class) {

                rawType = (Class<T>) type;

            } else if (type instanceof ParameterizedType) {

                ParameterizedType parameterizedType = (ParameterizedType) type;
                rawType = (Class<T>) parameterizedType.getRawType();

            } else if (type instanceof GenericArrayType) {

                rawType = (Class<T>) Object[].class;

            } else {
                throw new RuntimeException("Illegal type");
            }
        }

        return rawType;
    }

    /**
     * Return the direct child class that extends TypeLiteral<T>
     * @param clazz processed class
     */
    private static Class<?> getTypeLiteralSubclass(Class<?> clazz) {

        // Start with super class
        Class<?> superClass = clazz.getSuperclass();

        if (superClass.equals(TypeLiteral.class)) {
            // Super class is TypeLiteral, return the current class
            return clazz;
        } else if (superClass.equals(Object.class)) {
            // Hmm, strange case, we don not extends TypeLiteral !
            return null;
        } else {
            // Continue processing, one level deeper
            return (getTypeLiteralSubclass(superClass));
        }
    }

    /**
     * Return the value of the type parameter of TypeLiteral<T>.
     * @param typeLiteralSubclass subClass of TypeLiteral<T> to analyze
     * @return the parametrized type of TypeLiteral<T> (aka T)
     */
    private static Type getTypeParameter(Class<?> typeLiteralSubclass) {

        // Access the typeLiteral<T> super class using generics
        Type type = typeLiteralSubclass.getGenericSuperclass();
        if (type instanceof ParameterizedType) {
            // TypeLiteral is indeed parametrized
            ParameterizedType parameterizedType = (ParameterizedType) type;
            if (parameterizedType.getActualTypeArguments().length == 1) {
                // Return the value of the type parameter (aka T)
                return parameterizedType.getActualTypeArguments()[0];
            }
        }
        return null;
    }

    @Override
    public boolean equals(Object obj) {
        if (obj instanceof TypeLiteral<?>) {

            // Compare inner type for equality
            TypeLiteral<?> that = (TypeLiteral<?>) obj;
            return this.getType().equals(that.getType());
        }
        return false;
    }

    @Override
    public int hashCode() {
        return getType().hashCode();
    }

    @Override
    public String toString() {
        return getType().toString();
    }
}
