/**
 * OW2 Specifications
 * Copyright (C) 2007 Bull S.A.S.
 * Contact: easybeans@objectweb.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: Interaction.java 6201 2012-03-21 10:28:10Z benoitf $
 * --------------------------------------------------------------------------
 */

package javax.resource.cci;

import javax.resource.ResourceException;
 
/**
 * The javax.resource.cci.Interaction enables a component to execute EIS
 * functions. An Interaction instance supports the following ways of
 * interacting with an EIS instance:<ul>
 * <li>execute method that takes an input Record, output Record and an
 * InteractionSpec. This method executes the EIS function represented
 * by the InteractionSpec and updates the output Record</li>
 * <li>execute method that takes an input Record and an InteractionSpec.
 * This method implementation executes the EIS function represented by
 * the InteractionSpec and produces the output Record as a return value.</li>
 * 
 * An Interaction instance is created from a Connection and is required to
 * maintain its association with the Connection instance. The close method
 * releases all resources maintained by the resource adapter for the
 * Interaction. The close of an Interaction instance should not close the
 * associated Connection instance. 
 * @since 0.8
 * @version 0.8
 * @author Guillaume Sauthier
 */
public interface Interaction {

    /**
     * Closes the current Interaction and release all the resources held
     * for this instance by the resource adapter. The close of an Interaction
     * instance does not close the associated Connection instance. It is
     * recommended that Interaction instances be closed explicitly to free
     * any held resources.
     * @throws ResourceException Failed to close the Interaction instance.
     *         Invoking close on an already closed Interaction should also
     *         throw this exception.
     */
    void close() throws ResourceException;
    
    /**
     * Gets the Connection associated with the Interaction.
     * @return Connection instance associated with the Interaction
     */
    Connection getConnection();
    
    /**
     * Executes an interaction represented by the InteractionSpec. This form of
     * invocation takes an input Record and updates the output Record.
     * @param ispec InteractionSpec representing a target EIS data/function module
     * @param input Input Record
     * @param output Output Record
     * @return true if execution of the EIS function has been successful and
     *         output Record has been updated; false otherwise
     * @throws ResourceException Exception if execute operation fails. Examples
     *         of error cases are:<ul>
     *         <li>Resource adapter internal, EIS-specific or communication error</li>
     *         <li>Invalid specification of an InteractionSpec, input or output
     *         record structure</li>
     *         <li>Errors in use of input or output Record</li>
     *         <li>Invalid connection associated with this Interaction</li>
     *         </ul>
     * @throws NotSupportedException Operation not supported
     */
    boolean execute(InteractionSpec ispec,
                    Record input,
                    Record output) throws ResourceException;
    
    /**
     * Executes an interaction represented by the InteractionSpec. This form of
     * invocation takes an input Record and returns an output Record if the
     * execution of the Interaction has been successfull.
     * @param ispec InteractionSpec representing a target EIS data/function module
     * @param input Input Record
     * @return output Record if execution of the EIS function has been successful;
     *         null otherwise
     * @throws ResourceException Exception if execute operation fails. Examples
     *         of error cases are:<ul>
     *         <li>Resource adapter internal, EIS-specific or communication error</li>
     *         <li>Invalid specification of an InteractionSpec, input or output
     *         record structure</li>
     *         <li>Errors in use of input Record or creation of an output Record</li>
     *         <li>Invalid connection associated with this Interaction</li>
     *         </ul>
     * @throws NotSupportedException Operation not supported
     */
    Record execute(InteractionSpec ispec,
                   Record input) throws ResourceException;
    
    /**
     * Gets the first ResourceWarning from the chain of warnings associated with
     * this Interaction instance.
     * @return ResourceWarning at top of the warning chain
     * @throws ResourceException Failed to get ResourceWarnings associated with
     *         Interaction
     */
    ResourceWarning getWarnings() throws ResourceException;
    
    /**
     * Clears all the warning reported by this Interaction instance. After a call
     * to this method, the method getWarnings will return null until a new warning
     * is reported for this Interaction.
     * @throws ResourceException Failed to clear ResourceWarnings associated with
     *         Interaction
     */
    void clearWarnings() throws ResourceException;
}
