/**
 * OW2 Specifications
 * Copyright (C) 2007 Bull S.A.S.
 * Contact: easybeans@objectweb.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: LocalTransaction.java 6201 2012-03-21 10:28:10Z benoitf $
 * --------------------------------------------------------------------------
 */

package javax.resource.cci;

import javax.resource.ResourceException;
 
/**
 * The LocalTransaction defines a transaction demarcation interface for
 * resource manager local transactions. Note that this interface is
 * used for application level local transaction demarcation. The system
 * contract level LocalTransaction interface (as defined in the
 * javax.resource.spi package) is used by the container for local
 * transaction management.<br/>
 * 
 * A local transaction is managed internal to a resource manager.
 * There is no external transaction manager involved in the coordination
 * of such transactions.<br/>
 * 
 * A CCI implementation can (but is not required to) implement the
 * LocalTransaction interface. If the LocalTransaction interface is
 * supported by a CCI implementation, then the method
 * Connection.getLocalTransaction should return a LocalTransaction
 * instance. A component can then use the returned LocalTransaction
 * to demarcate a resource manager local transaction (associated
 * with the Connection instance) on the underlying EIS instance.
 * @since 0.8
 * @author Guillaume Sauthier
 */
public interface LocalTransaction {

    /**
     * Begins a local transaction on an EIS instance.
     * @throws ResourceException Failed to begin a local transaction.
     *         Examples of error cases are:<ul>
     *         <li>Resource adapter internal or EIS-specific error</li>
     *         <li>Connection is already participating in a local or JTA
     *         transaction</li>
     *         </ul> 
     */
    void begin() throws ResourceException;
    
    /**
     * Commits the current local transaction and release all locks held
     * by the underlying EIS instance.
     * @throws ResourceException Failed to commit a local transaction.
     *         Examples of error cases are:<ul>
     *         <li>Resource adapter internal or EIS-specific error</li>
     *         <li>Violation of integrity constraints, deadlock detection,
     *         communication failure during transaction completion, or any
     *         retry requirement</li>
     *         <li>Connection is participating in an active JTA transaction</li>
     *         <li>Invalid transaction context; commit operation invoked
     *         without an active transaction context</li>
     *         </ul> 
     */
    void commit() throws ResourceException;
    
    /**
     * Rollbacks the current resource manager local transaction.
     * @throws ResourceException Failed to rollback a local transaction.
     *         Examples of error cases are:<ul>
     *         <li>Resource adapter internal or EIS-specific error</li>
     *         <li>Connection is participating in an active JTA transaction</li>
     *         <li>Invalid transaction context; rollback operation invoked
     *         without an active transaction context</li>
     *         </ul> 
     */
    void rollback() throws ResourceException;
}
