/**
 * OW2 Specifications
 * Copyright (C) 2007 Bull S.A.S.
 * Contact: easybeans@objectweb.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: ConnectionEvent.java 6201 2012-03-21 10:28:10Z benoitf $
 * --------------------------------------------------------------------------
 */
 
package javax.resource.spi;

import java.util.EventObject;

/**
 * The ConnectionEvent class provides information about the source of a
 * connection related event.A ConnectionEvent instance contains the
 * following information:<br>
 * <ul>
 *   <li>Type of the connection event</li>
 *   <li>ManagedConnection instance that generated the connection event.
 *   A ManagedConnection instance is returned from the method
 *   ConnectionEvent.getSource.</li>
 *   <li>Connection handle associated with the ManagedConnection
 *   instance; required for the CONNECTION_CLOSED event and optional
 *   for the other event types.</li>
 *   <li>Optionally, an exception indicating the connection related
 *   error. Note that exception is used for CONNECTION_ERROR_OCCURRED.</li>
 * </ul>
 * This class defines following types of event notifications:<br/>
 * <ul>
 *   <li>CONNECTION_CLOSED</li>
 *   <li>LOCAL_TRANSACTION_STARTED</li>
 *   <li>LOCAL_TRANSACTION_COMMITTED</li>
 *   <li>LOCAL_TRANSACTION_ROLLEDBACK</li>
 *   <li>CONNECTION_ERROR_OCCURRED</li>
 * </ul>
 * @version 0.5 
 * @author Guillaume Sauthier
 */
public class ConnectionEvent extends EventObject {

    /**
     * Serial version UID.
     */
    private static final long serialVersionUID = 5611772461379563249L;

    /**
     * Event notification that an application component has closed
     * the connection.
     */
    public static final int CONNECTION_CLOSED = 1;
    
    /**
     * Event notification that a Resource Manager Local Transaction
     * was started on the connection.
     */
    public static final int LOCAL_TRANSACTION_STARTED = 2;
    
    /**
     * Event notification that a Resource Manager Local Transaction
     * was committed on the connection
     */
    public static final int LOCAL_TRANSACTION_COMMITTED = 3;
    
    /**
     * Event notification that a Resource Manager Local Transaction
     * was rolled back on the connection
     */
    public static final int LOCAL_TRANSACTION_ROLLEDBACK = 4;
    
    /**
     * Event notification that an error occurred on the connection.
     * This event indicates that the ManagedConnection instance is
     * now invalid and unusable.
     */
    public static final int CONNECTION_ERROR_OCCURRED = 5;
    
    
    /**
     * Type of the event.
     */
    protected int id;
    
    /**
     * Exception about to be thrown to the application.
     */
    private Exception exception;

    /**
     * Connection handle.
     */
    private Object connectionHandle;
    
    /**
     * Construct a ConnectionEvent object. Exception defaults to null.
     * @param source ManagedConnection that is the source of the event
     * @param eid type of the Connection event
     */
    public ConnectionEvent(ManagedConnection source,
                           int eid) {
        this(source, eid, null);
    }

    /**
     * Construct a ConnectionEvent object.
     * @param source ManagedConnection that is the source of the event
     * @param eid type of the Connection event
     * @param exception exception about to be thrown to the application
     */
    public ConnectionEvent(ManagedConnection source,
                           int eid,
                           Exception exception) {
        super(source);
        this.id = eid;
        this.exception = exception;
    }
    
    /**
     * Get the connection handle associated with the Managed Connection
     * instance. Used for CONNECTION_CLOSED event.
     * @return the connection handle. May be null
     */
    public Object getConnectionHandle() {
        return connectionHandle;
    }
    
    /**
     * Set the connection handle. Used for CONNECTION_CLOSED event
     * @param connectionHandle
     */
    public void setConnectionHandle(Object connectionHandle) {
        this.connectionHandle = connectionHandle;
    }
    
    /**
     * Get the exception. May be null.
     * @return the exception about to be thrown.
     */
    public Exception getException() {
        return exception;
    }
    
    /**
     * Get the type of event
     * @return the type of the event
     */
    public int getId() {
        return this.id;
    }
}
