/**
 * OW2 Specifications
 * Copyright (C) 2007 Bull S.A.S.
 * Contact: easybeans@objectweb.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: MessageEndpoint.java 6201 2012-03-21 10:28:10Z benoitf $
 * --------------------------------------------------------------------------
 */

package javax.resource.spi.endpoint;

import java.lang.reflect.Method;

import javax.resource.ResourceException;

/**
 * This defines a contract for a message endpoint. This is implemented
 * by an application server.
 * @version 1.0
 * @author Guillaume Sauthier
 */
public interface MessageEndpoint {

    /**
     * This is called by a resource adapter before a message is delivered.
     * @param method description of a target method. This information
     *        about the intended target method allows an application
     *        server to decide whether to start a transaction during
     *        this method call, depending on the transaction preferences
     *        of the target method. The processing (by the application
     *        server) of the actual message delivery method call on the
     *        endpoint must be independent of the class loader associated
     *        with this descriptive method object.
     * @throws NoSuchMethodException indicates that the specified method
     *         does not exist on the target endpoint.
     * @throws ResourceException generic exception.
     * @throws ApplicationServerInternalException indicates an error condition
     *         in the application server.
     * @throws IllegalStateException indicates that the endpoint is in an
     *         illegal state for the method invocation. For example, this
     *         occurs when beforeDelivery and afterDelivery method calls
     *         are not paired.
     * @throws UnavailableException indicates that the endpoint is not
     *         available.
     */
    void beforeDelivery(Method method) throws NoSuchMethodException,
                                              ResourceException;

    /**
     * This is called by a resource adapter after a message is delivered.
     * @throws ResourceException generic exception.
     * @throws ApplicationServerInternalException indicates an error
     *         condition in the application server.
     * @throws IllegalStateException indicates that the endpoint is
     *         in an illegal state for the method invocation. For
     *         example, this occurs when beforeDelivery and afterDelivery
     *         method calls are not paired.
     * @throws UnavailableException indicates that the endpoint is not
     *         available.
     */
    void afterDelivery() throws ResourceException;

    /**
     * This method may be called by the resource adapter to indicate that
     * it no longer needs a proxy endpoint instance. This hint may be used
     * by the application server for endpoint pooling decisions.
     */
    void release();
}
