/**
 * EasyBeans
 * Copyright (C) 2006 Bull S.A.S.
 * Contact: easybeans@objectweb.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: Query.java 4359 2008-12-10 10:24:40Z sauthieg $
 * --------------------------------------------------------------------------
 */

package javax.persistence;

import java.util.Calendar;
import java.util.Date;
import java.util.List;

/**
 * Interface used to control query execution.
 * @see <a href="http://www.jcp.org/en/jsr/detail?id=220">EJB 3.0 specification</a>
 * @author Florent Benoit
 * @since EJB 3.0 version.
 */

public interface Query {
    /**
     * Execute a SELECT query and return the query results as a List.
     * @return a list of the results
     * @throws IllegalStateException if called for a Java Persistence query
     *         language UPDATE or DELETE statement
     */
    List getResultList();

    /**
     * Execute a SELECT query that returns a single result.
     * @return the result
     * @throws NoResultException if there is no result
     * @throws NonUniqueResultException if more than one result
     * @throws IllegalStateException if called for a Java Persistence query
     *         language UPDATE or DELETE statement
     */
    Object getSingleResult();

    /**
     * Execute an update or delete statement.
     * @return the number of entities updated or deleted
     * @throws IllegalStateException if called for a Java Persistence query
     *         language SELECT statement
     * @throws TransactionRequiredException if there is no transaction
     */
    int executeUpdate();

    /**
     * Set the maximum number of results to retrieve.
     * @param maxResult the given max number.
     * @return the same query instance
     * @throws IllegalArgumentException if argument is negative
     */
    Query setMaxResults(int maxResult);

    /**
     * Set the position of the first result to retrieve.
     * @param startPosition position of the first result, numbered from 0
     * @return the same query instance
     * @throws IllegalArgumentException if argument is negative
     */
    Query setFirstResult(int startPosition);

    /**
     * Set an implementation-specific hint. If the hint name is not recognized,
     * it is silently ignored.
     * @param hintName the name of the hint.
     * @param value the value of the hint.
     * @return the same query instance
     * @throws IllegalArgumentException if the second argument is not valid for
     *         the implementation
     */
    Query setHint(String hintName, Object value);

    /**
     * Bind an argument to a named parameter.
     * @param name the parameter name
     * @param value the value of the parameter.
     * @return the same query instance
     * @throws IllegalArgumentException if parameter name does not correspond to
     *         parameter in query string or argument is of incorrect type
     */
    Query setParameter(String name, Object value);

    /**
     * Bind an instance of java.util.Date to a named parameter.
     * @param name the name of the parameter.
     * @param value the value of the parameter.
     * @param temporalType the type of temporal.
     * @return the same query instance
     * @throws IllegalArgumentException if parameter name does not correspond to
     *         parameter in query string
     */
    Query setParameter(String name, Date value, TemporalType temporalType);

    /**
     * Bind an instance of java.util.Calendar to a named parameter.
     * @param name the name of the parameter.
     * @param value the value of the parameter.
     * @param temporalType the type of temporal.
     * @return the same query instance
     * @throws IllegalArgumentException if parameter name does not correspond to
     *         parameter in query string
     */
    Query setParameter(String name, Calendar value, TemporalType temporalType);

    /**
     * Bind an argument to a positional parameter.
     * @param position the position parameter.
     * @param value the given value.
     * @return the same query instance
     * @throws IllegalArgumentException if position does not correspond to
     *         positional parameter of query or argument is of incorrect type
     */
    Query setParameter(int position, Object value);

    /**
     * Bind an instance of java.util.Date to a positional parameter.
     * @param position the position parameter.
     * @param value the given value.
     * @param temporalType the type of temporal.
     * @return the same query instance
     * @throws IllegalArgumentException if position does not correspond to
     *         positional parameter of query
     */
    Query setParameter(int position, Date value, TemporalType temporalType);

    /**
     * Bind an instance of java.util.Calendar to a positional parameter.
     * @param position the position parameter.
     * @param value the given value.
     * @param temporalType the type of temporal.
     * @return the same query instance
     * @throws IllegalArgumentException if position does not correspond to
     *         positional parameter of query
     */
    Query setParameter(int position, Calendar value, TemporalType temporalType);

    /**
     * Set the flush mode type to be used for the query execution. The flush
     * mode type applies to the query regardless of the flush mode type in use
     * for the entity manager.
     * @return the query.
     * @param flushMode the flush mode.
     */
    Query setFlushMode(FlushModeType flushMode);

}
