/**
 * OW2 Specifications
 * Copyright (C) 2010 Bull S.A.S.
 * Contact: easybeans@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: TableGenerator.java 5417 2010-03-24 09:04:14Z benoitf $
 * --------------------------------------------------------------------------
 */

package javax.persistence;

import static java.lang.annotation.ElementType.FIELD;
import static java.lang.annotation.ElementType.METHOD;
import static java.lang.annotation.ElementType.TYPE;
import static java.lang.annotation.RetentionPolicy.RUNTIME;

import java.lang.annotation.Retention;
import java.lang.annotation.Target;

/**
 * The TableGenerator annotation defines a primary key generator that may be
 * referenced by name when a generator element is specified for the
 * GeneratedValue annotation. A table generator may be specified on the entity
 * class or on the primary key field or property. The scope of the generator
 * name is global to the persistence unit (across all generator types).
 * @see <a href="http://jcp.org/en/jsr/detail?id=317">JPA 2.0 specification</a>
 * @author Florent Benoit
 * @since JPA 1.0 version.
 */
@Target({ TYPE, METHOD, FIELD })
@Retention(RUNTIME)
public @interface TableGenerator {

    /**
     * (Required) A unique generator name that can be referenced by one or more
     * classes to be the generator for id values.
     */
    String name();

    /**
     * (Optional) Name of table that stores the generated id values.
     */
    String table() default "";

    /**
     * (Optional) The catalog of the table.
     */
    String catalog() default "";

    /**
     * (Optional) The schema of the table.
     */
    String schema() default "";

    /**
     * (Optional) Name of the primary key column in the table.
     */
    String pkColumnName() default "";

    /**
     * (Optional) Name of the column that stores the last value generated.
     */
    String valueColumnName() default "";

    /**
     * (Optional) The primary key value in the generator table that
     * distinguishes this set of generated values from others that may be stored
     * in the table.
     */
    String pkColumnValue() default "";

    /**
     * (Optional) The value used to initialize the column that stores the last
     * value generated.
     */
    int initialValue() default 0;

    /**
     * (Optional) The amount to increment by when allocating id numbers from the
     * generator.
     */
    int allocationSize() default 50;

    /**
     * (Optional) Unique constraints that are to be placed on the table. These
     * are only used if table generation is in effect. These constraints apply
     * in addition to primary key constraints .
     */
    UniqueConstraint[] uniqueConstraints() default {};

}
