/**
 * OW2 Specifications
 * Copyright (C) 2007 Bull S.A.S.
 * Contact: easybeans@objectweb.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: UserTransaction.java 4359 2008-12-10 10:24:40Z sauthieg $
 * --------------------------------------------------------------------------
 */

package javax.transaction;

/** 
 * The UserTransaction interface defines the methods that allow an application
 * to explicitly manage transaction boundaries.
 * 
 * @author Guillaume Sauthier
 */
public interface UserTransaction {

    /**
     * Create a new transaction and associate it with the current thread.
     * @throws NotSupportedException Thrown if the thread is already
     *         associated with a transaction and the Transaction Manager
     *         implementation does not support nested transactions.
     * @throws SystemException Thrown if the transaction manager encounters
     *         an unexpected error condition.
     */
    void begin() throws NotSupportedException, SystemException;

    /**
     * Complete the transaction associated with the current thread. When
     * this method completes, the thread is no longer associated with a
     * transaction.
     * @throws RollbackException Thrown to indicate that the transaction
     *         has been rolled back rather than committed.
     * @throws HeuristicMixedException Thrown to indicate that a heuristic
     *         decision was made and that some relevant updates have been
     *         committed while others have been rolled back.
     * @throws HeuristicRollbackExceptionThrown to indicate that a heuristic
     *         decision was made and that all relevant updates have been
     *         rolled back.
     * @throws SecurityException Thrown to indicate that the thread is not
     *         allowed to commit the transaction.
     * @throws IllegalStateException Thrown if the current thread is not
     *         associated with a transaction.
     * @throws SystemException Thrown if the transaction manager encounters
     *         an unexpected error condition.
     */
    void commit() throws RollbackException,
                         HeuristicMixedException,
                         HeuristicRollbackException,
                         SecurityException,
                         IllegalStateException,
                         SystemException;
    
    /**
     * Roll back the transaction associated with the current thread. When
     * this method completes, the thread is no longer associated with a
     * transaction.
     * @throws IllegalStateException Thrown if the current thread is not
     *         associated with a transaction.
     * @throws SecurityException Thrown to indicate that the thread is not
     *         allowed to roll back the transaction.
     * @throws SystemException Thrown if the transaction manager encounters
     *         an unexpected error condition.
     */
    void rollback() throws IllegalStateException,
                           SecurityException,
                           SystemException;

    /**
     * Modify the transaction associated with the current thread such that
     * the only possible outcome of the transaction is to roll back the transaction.
     * @throws IllegalStateException Thrown if the current thread is not
     *         associated with a transaction.
     * @throws SystemException Thrown if the transaction manager encounters
     *         an unexpected error condition.
     */
    void setRollbackOnly() throws IllegalStateException,
                                  SystemException;

    /**
     * Obtain the status of the transaction associated with the current thread.
     * @return The transaction status. If no transaction is associated with
     *         the current thread, this method returns the Status.NoTransaction value.
     * @throws SystemException Thrown if the transaction manager encounters
     *         an unexpected error condition.
     */
    int getStatus() throws SystemException;

    /**
     * Modify the timeout value that is associated with transactions started
     * by the current thread with the begin method.<br/>
     * 
     * If an application has not called this method, the transaction
     * service uses some default value for the transaction timeout. 
     * @param seconds The value of the timeout in seconds. If the value is
     *        zero, the transaction service restores the default value.
     *        If the value is negative a SystemException is thrown.
     * @throws SystemException Thrown if the transaction manager encounters
     *         an unexpected error condition.
     */
    void setTransactionTimeout(int seconds) throws SystemException;
}
