/**
 * OW2 Spec
 * Copyright (C) 2010 Bull S.A.S.
 * Contact: easybeans@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: Scope.java 5374 2010-02-25 17:22:19Z sauthieg $
 * --------------------------------------------------------------------------
 */

package javax.inject;

import java.lang.annotation.Documented;
import java.lang.annotation.Retention;
import java.lang.annotation.Target;

import static java.lang.annotation.ElementType.ANNOTATION_TYPE;
import static java.lang.annotation.RetentionPolicy.RUNTIME;

/**
 * Identifies scope annotations. A scope annotation applies to a class containing an injectable
 * constructor and governs how the injector reuses instances of the type. By default, if no
 * scope annotation is present, the injector creates an instance (by injecting the type's constructor),
 * uses the instance for one injection, and then forgets it. If a scope annotation is present, the
 * injector may retain the instance for possible reuse in a later injection. If multiple threads can
 * access a scoped instance, its implementation should be thread safe. The implementation of the scope
 * itself is left up to the injector.
 *
 * In the following example, the scope annotation <code>@Singleton</code> ensures that we only have one Log instance:
 * <pre>
 *   &#064;Singleton
 *   class Log {
 *     void log(String message) { ... }
 *   }
 * </pre>
 *
 * The injector generates an error if it encounters more than one scope annotation on the same class
 * or a scope annotation it doesn't support.
 *
 * A scope annotation:
 * <ul>
 *  <li>is annotated with @Scope, @Retention(RUNTIME), and typically @Documented.</li>
 *  <li>should not have attributes.</li>
 *  <li>is typically not @Inherited, so scoping is orthogonal to implementation inheritance.</li>
 *  <li>may have restricted usage if annotated with @Target. While this specification covers applying
 *      scopes to classes only, some injector configurations might use scope annotations in other places
 *      (on factory method results for example).</li>
 * </ul>
 *
 * For example:
 * <pre>
 *  &#064;Documented
 *  &#064;Retention(RUNTIME)
 *  &#064;Scope
 *  public @interface RequestScoped {}
 * </pre>
 *
 * Annotating scope annotations with <code>@Scope</scope> helps the injector detect the case where
 * a programmer used the scope annotation on a class but forgot to configure the scope in the injector.
 * A conservative injector would generate an error rather than not apply a scope.
 *
 * @see javax.inject.Singleton
 * @author Guillaume Sauthier
 * @since DI 1.0
 */
@Target(ANNOTATION_TYPE)
@Retention(RUNTIME)
@Documented
public @interface Scope {}