/**
 * OW2 Spec
 * Copyright (C) 2010 Bull S.A.S.
 * Contact: easybeans@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: Conversation.java 5375 2010-02-25 17:25:09Z sauthieg $
 * --------------------------------------------------------------------------
 */

package javax.enterprise.context;

/**
 * Allows the application to manage the conversation context by marking the current
 * conversation as transient or long-running, specifying a conversation identifier,
 * or setting the conversation timeout.
 *
 * An instance may be injected:
 * <pre>
 *  &#064;Inject Conversation conversation;
 * </pre>
 *
 * The conversation timeout is a hint to the container that a conversation should not
 * be destroyed if it has been active within the last given interval in milliseconds.
 *
 * @author Guillaume Sauthier
 * @see javax.enterprise.context.ConversationScoped
 */
public interface Conversation {

    /**
     * Mark the current transient conversation long-running. A conversation identifier
     * is generated by the container.
     *
     * @throws IllegalStateException if the current conversation is already marked long-running.
     */
    void begin();

    /**
     * Mark the current transient conversation long-running, with a specified identifier.
     *
     * @param id Conversation identifier
     * @throws IllegalStateException if the current conversation is already marked long-running. 
     * @throws IllegalArgumentException if a conversation with the specified identifier already exists.
     */
    void begin(String id);

    /**
     * Marks the current long-running conversation transient.
     * @throws IllegalStateException if the current conversation is already marked transient.
     */
    void end();

    /**
     * Get the identifier of the current long-running conversation.
     *
     * @return the identifier of the current long-running conversation, or a null value
     * if the current conversation is transient.
     */
    String getId();

    /**
     * Get the timeout of the current conversation.
     *
     * @return the current timeout in milliseconds.
     */
    long getTimeout();

    /**
     * Set the timeout of the current conversation.
     *
     * @param milliseconds the new timeout in milliseconds.
     */
    void setTimeout(long milliseconds);

    /**
     * Determine if the conversation is marked transient or long-running.
     *
     * @return <tt>true</tt> if the conversation is marked transient, or <tt>false</tt> if
     * it is marked long-running.
     */
    boolean isTransient();
    
}
