/**
 * OW2 Specifications
 * Copyright (C) 2007 Bull S.A.S.
 * Contact: easybeans@objectweb.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: ResultSetInfo.java 4359 2008-12-10 10:24:40Z sauthieg $
 * --------------------------------------------------------------------------
 */

package javax.resource.cci;

import javax.resource.ResourceException;
 
/**
 * The interface javax.resource.cci.ResultSetInfo provides information 
 * the support provided for ResultSet by a connected EIS instance. A
 * component calls the method Connection.getResultInfo to get the
 * ResultSetInfo instance.<br/>
 * 
 * A CCI implementation is not required to support javax.resource.cci.ResultSetInfo
 * interface. The implementation of this interface is provided only if
 * the CCI supports the ResultSet facility.
 * @version 0.9 
 * @author Guillaume Sauthier
 */
public interface ResultSetInfo {

    /**
     * Indicates whether or not a visible row update can be detected
     * by calling the method ResultSet.rowUpdated.
     * @param type type of the ResultSet i.e. ResultSet.TYPE_XXX
     * @return true if changes are detected by the result set type;
     *        false otherwise
     * @throws ResourceException Failed to get the information
     */
    boolean updatesAreDetected(int type) throws ResourceException;
    
    /**
     * Indicates whether or not a visible row insert can be detected
     * by calling ResultSet.rowInserted.
     * @param type type of the ResultSet i.e. ResultSet.TYPE_XXX
     * @return true if changes are detected by the result set type;
     *         false otherwise
     * @throws ResourceException Failed to get the information
     */
    boolean insertsAreDetected(int type) throws ResourceException;
    
    boolean deletesAreDetected(int type) throws ResourceException;
    
    /**
     * Indicates whether or not a resource adapter supports a type
     * of ResultSet.
     * @param type type of the ResultSet i.e. ResultSet.TYPE_XXX
     * @return true if ResultSet type supported; false otherwise
     * @throws ResourceException Failed to get the information
     */
    boolean supportsResultSetType(int type) throws ResourceException;
    
    /**
     * Indicates whether or not a resource adapter supports the
     * concurrency type in combination with the given ResultSet
     * type.
     * @param type type of the ResultSet i.e. ResultSet.TYPE_XXX
     * @param concurrency ResultSet concurrency type defined in
     *        java.sql.ResultSet
     * @return true if the specified combination supported; false
     *         otherwise
     * @throws ResourceException Failed to get the information
     */
    boolean supportsResultTypeConcurrency(int type,
                                          int concurrency)
            throws ResourceException;
    
    /**
     * Indicates whether updates made by others are visible.
     * @param type type of the ResultSet i.e. ResultSet.TYPE_XXX
     * @return true if updates by others are visible for the
     *         ResultSet type; false otherwise
     * @throws ResourceException Failed to get the information
     */
    boolean othersUpdatesAreVisible(int type) throws ResourceException;
    
    /**
     * Indicates whether deletes made by others are visible.
     * @param type type of the ResultSet i.e. ResultSet.TYPE_XXX
     * @return true if deletes by others are visible for the ResultSet
     *         type; false otherwise
     * @throws ResourceException Failed to get the information
     */
    boolean othersDeletesAreVisible(int type) throws ResourceException;
    
    /**
     * Indicates whether inserts made by others are visible.
     * @param type type of the ResultSet i.e. ResultSet.TYPE_XXX
     * @return true if inserts by others are visible for the
     *         ResultSet type; false otherwise
     * @throws ResourceException Failed to get the information
     */
    boolean othersInsertsAreVisible(int type) throws ResourceException;
    
    boolean ownUpdatesAreVisible(int type) throws ResourceException;
    
    boolean ownInsertsAreVisible(int type) throws ResourceException;
    
    boolean ownDeletesAreVisible(int type) throws ResourceException;
}
