/**
 * OW2 Specifications
 * Copyright (C) 2007 Bull S.A.S.
 * Contact: easybeans@objectweb.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: Connection.java 6201 2012-03-21 10:28:10Z benoitf $
 * --------------------------------------------------------------------------
 */

package javax.resource.cci;

import javax.resource.ResourceException;

/**
 * A Connection represents an application-level handle that is used
 * by a client to access the underlying physical connection. The
 * actual physical connection associated with a Connection instance
 * is represented by a ManagedConnection instance.<br/>
 *
 * A client gets a Connection instance by using the <code>getConnection</code>
 * method on a {@link ConnectionFactory} instance. A connection can be
 * associated with zero or more {@link Interaction} instances.
 * @author Guillaume Sauthier
 */
public interface Connection {

    /**
     * Creates an Interaction associated with this Connection. An Interaction
     * enables an application to execute EIS functions.
     * @return Interaction instance
     * @throws ResourceException Failed to create an Interaction
     */
    Interaction createInteraction() throws ResourceException;

    /**
     * Returns an LocalTransaction instance that enables a component to
     * demarcate resource manager local transactions on the Connection.
     * If a resource adapter does not allow a component to demarcate
     * local transactions on an Connection using LocalTransaction interface,
     * then the method getLocalTransaction should throw a NotSupportedException.
     * @return LocalTransaction instance
     * @throws ResourceException Failed to return a LocalTransaction instance
     *         because of a resource adapter error
     * @throws NotSupportedException Demarcation of Resource manager local
     *         transactions is not supported on this Connection
     */
    LocalTransaction getLocalTransaction() throws ResourceException;

    /**
     * Gets the information on the underlying EIS instance represented
     * through an active connection.
     * @return ConnectionMetaData instance representing information about
     *         the EIS instance
     * @throws ResourceException Failed to get information about the connected
     *         EIS instance. Error can be resource adapter-internal,
     *         EIS-specific or communication related.
     */
    ConnectionMetaData getMetaData() throws ResourceException;

    /**
     * Gets the information on the ResultSet functionality supported by a
     * connected EIS instance.
     * @return ResultSetInfo instance
     * @throws ResourceException Failed to get ResultSet related information
     * @throws NotSupportedException ResultSet functionality is not supported
     */
    ResultSetInfo getResultSetInfo() throws ResourceException;

    /**
     * Initiates close of the connection handle at the application level. A
     * client should not use a closed connection to interact with an EIS.
     * @throws ResourceException Exception thrown if close on a connection
     *         handle fails.<br/>Any invalid connection close invocation--example,
     *         calling close on a connection handle that is already closed--should
     *         also throw this exception.
     */
    void close() throws ResourceException;
}
