/**
 * OW2 Specifications
 * Copyright (C) 2007 Bull S.A.S.
 * Contact: easybeans@objectweb.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: MessageEndpointFactory.java 6201 2012-03-21 10:28:10Z benoitf $
 * --------------------------------------------------------------------------
 */

package javax.resource.spi.endpoint;

import java.lang.reflect.Method;

import javax.resource.spi.UnavailableException;
import javax.transaction.xa.XAResource;

/**
 * This serves as a factory for creating message endpoints.
 * @version 1.0
 * @author Guillaume Sauthier
 */
public interface MessageEndpointFactory {

    /**
     * This is used to create a message endpoint. The message endpoint
     * is expected to implement the correct message listener type.
     * @param xaResource an optional XAResource instance used to get
     *        transaction notifications when the message delivery is
     *        transacted.
     * @return a message endpoint instance.
     * @throws UnavailableException indicates a transient failure in
     *         creating a message endpoint. Subsequent attempts to
     *         create a message endpoint might succeed.
     */
    MessageEndpoint createEndpoint(XAResource xaResource) throws UnavailableException;
    
    /**
     * This is used to create a message endpoint. The message endpoint is expected to implement the correct message listener type.
     * @param xaResource an optional XAResource instance used to get transaction notifications when the message delivery is
     * transacted.
     * @param timeout an optional value used to specify the time duration (in milliseconds) within which the message endpoint
     * needs to be created by the MessageEndpointFactory. Otherwise, the MessageEndpointFactory rejects the creation of the
     * MessageEndpoint with an UnavailableException. Note, this does not offer real-time guarantees.
     * @return a message endpoint instance.
     * @throws UnavailableException indicates a transient failure in creating a message endpoint. Subsequent attempts to create a
     * message endpoint might succeed.
     * @since 1.6
     */
    MessageEndpoint createEndpoint(XAResource xaResource, long timeout) throws UnavailableException;

    /**
     * This is used to find out whether message deliveries to a target
     * method on a message listener interface that is implemented by a
     * message endpoint will be transacted or not. The message endpoint
     * may indicate its transacted delivery preferences (at a per method
     * level) through its deployment descriptor. The message delivery
     * preferences must not change during the lifetime of a message endpoint.
     * @param method description of a target method. This information about
     *        the intended target method allows an application server to
     *        find out whether the target method call will be transacted
     *        or not.
     * @return true, if message endpoint requires transacted message delivery.
     * @throws NoSuchMethodException indicates that the specified method does
     *         not exist on the target endpoint.
     */
    boolean isDeliveryTransacted(Method method) throws NoSuchMethodException;
}
