/**
 * OW2 Specifications
 * Copyright (C) 2007 Bull S.A.S.
 * Contact: easybeans@objectweb.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: ProgressObject.java 4359 2008-12-10 10:24:40Z sauthieg $
 * --------------------------------------------------------------------------
 */

package javax.enterprise.deploy.spi.status;

import javax.enterprise.deploy.spi.TargetModuleID;
import javax.enterprise.deploy.spi.exceptions.OperationUnsupportedException;

/**
 * The ProgressObject interface tracks and reports the progress of the deployment
 * activities, distribute, start, stop, undeploy.<br/>
 *
 * This class has an optional cancel method. The support of the cancel function
 * can be tested by the isCancelSupported method.<br/>
 *
 * The ProgressObject structure allows the user the option of polling for status
 * or to provide a callback.
 * @author Guillaume Sauthier
 */
public interface ProgressObject {

    /**
     * Retrieve the status of this activity.
     * @return An object containing the status information.
     */
    DeploymentStatus getDeploymentStatus();

    /**
     * Retrieve the list of TargetModuleIDs successfully processed or created
     * by the associated DeploymentManager operation.
     * @return a list of TargetModuleIDs.
     */
    TargetModuleID[] getResultTargetModuleIDs();

    /**
     * Return the ClientConfiguration object associated with the TargetModuleID.
     * @param id represents the client module
     * @return ClientConfiguration for a given TargetModuleID or null if none exists.
     */
    ClientConfiguration getClientConfiguration(TargetModuleID id);

    /**
     * Tests whether the vendor supports a cancel opertation for deployment activities.
     * @return true if canceling an activity is supported by this platform.
     */
    boolean isCancelSupported();

    /**
     * (optional) A cancel request on an in-process operation stops all further
     * processing of the operation and returns the environment to it original
     * state before the operation was executed. An operation that has run to
     * completion cannot be canceled.
     * @throws OperationUnsupportedException this optional command is not supported
     *         by this implementation.
     */
    void cancel() throws OperationUnsupportedException;

    /**
     * Tests whether the vendor supports a stop operation for deployment activities.
     * @return true if canceling an activity is supported by this platform.
     */
    boolean isStopSupported();

    /**
     * (optional) A stop request on an in-process operation allows the operation on
     * the current TargetModuleID to run to completion but does not process any of
     * the remaining unprocessed TargetModuleID objects. The processed TargetModuleIDs
     * must be returned by the method getResultTargetModuleIDs.
     * @throws OperationUnsupportedException this optional command is not supported
     *         by this implementation.
     */
    void stop() throws OperationUnsupportedException;

    /**
     * Add a listener to receive Progress events on deployment actions.
     * @param pol the listener to receive events
     */
    void addProgressListener(ProgressListener pol);

    /**
     * Remove a ProgressObject listener.
     * @param pol the listener being removed
     */
    void removeProgressListener(ProgressListener pol);
}
