/**
 * EasyBeans
 * Copyright (C) 2010 Bull S.A.S.
 * Contact: easybeans@objectweb.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: MessagePolicy.java 6201 2012-03-21 10:28:10Z benoitf $
 * --------------------------------------------------------------------------
 */
package javax.security.auth.message;

/**
 * This class defines a message authentication policy.
 *
 * <p />
 *
 * A ClientAuthContext uses this class to communicate
 * (at module initialization time) request and response message protection
 * policies to its ClientAuthModule objects.
 * A ServerAuthContext uses this class to communicate
 * request and response message protection
 * policies to its ServerAuthModule objects.
 *
 * @version 1.0
 * @see ClientAuthContext
 * @see ServerAuthContext
 * @see ClientAuthModule
 * @see ServerAuthModule
 */
public class MessagePolicy {

    /**
     * The array of target policies of the message policy.
     */
    private TargetPolicy[] targetPolicies;

    /**
     * The mandatory status of the policy. True if the message policy is
     * required, False if it is optional.
     */
    private boolean isMandatory;

    /**
     * Create a MessagePolicy instance with an array of target policies.
     *
     * @param targetPolicies an array of target policies.
     *
     * @param mandatory A boolean value indicating whether the MessagePolicy is
     *          mandatory or optional.
     *
     * @exception IllegalArgumentException if the specified targetPolicies
     *          is null.
     */
    public MessagePolicy(TargetPolicy[] targetPolicies, boolean mandatory) {
        if (targetPolicies == null) {
            throw new IllegalArgumentException("targetPolicies argument cannot be null");
        }
        this.targetPolicies = (TargetPolicy[]) targetPolicies.clone();
        this.isMandatory = mandatory;
    }

    /**
     * Get the MessagePolicy modifier.
     *
     * @return A boolean indicating whether the MessagePolicy is
     *         optional(false) or required(true).
     */
    public boolean isMandatory() {
        return this.isMandatory;
    }

    /**
     * Get the target policies that comprise the authentication policy.
     *
     * <p />
     *
     * @return An array of target authentication policies, where each element
     * describes an authentication policy and the parts of the message to which
     * the authentication policy applies. This method returns null to indicate
     * that no security operations should be performed on the messages to which
     * the policy applies. This method never returns a zero-length array.
     *
     * <p />
     *
     * When this method returns an array of target policies, the
     * order of elements in the array represents the order that the
     * corresponding message transformations or validations described
     * by the target policies are to be performed by the authentication
     * module.
     */
    public TargetPolicy[] getTargetPolicies() {
        return (TargetPolicy[]) this.targetPolicies.clone();
    }

    /**
     * This class defines the message protection policies for specific Targets.
     *
     * <p />
     *
     * This class is used to associate a message protection policy with
     * targets within a message. Message targets are represented using
     * an implementation of the <i>Target</i> interface matched to the
     * message types in MessageInfo. The message protection policy is
     * identified by an implementation of the <i>ProtectionPolicy</i>
     * interface.
     *
     * @see ClientAuthModule
     * @see ServerAuthModule
     */
    public static class TargetPolicy {
        /**
         * The targets associated to a protection policy.
         */
        private Target[] targets;

        /**
         * The protection policy associated to the targets.
         */
        private ProtectionPolicy policy;

        /**
         * Create a TargetPolicy instance with an array of Targets
         * and with a ProtectionPolicy.
         *
         * @param targets An array of Targets. This argument may be null.
         *
         * @param protectionPolicy The object that describes
         * the message authentication policy that applies to the
         * targets.
         *
         * @exception IllegalArgumentException if the specified targets
         *		or protectionPolicy is null.
         */
        public TargetPolicy(Target[] targets,
                ProtectionPolicy protectionPolicy) {
            if (protectionPolicy == null) {
                throw new IllegalArgumentException("protectionPolicy argument cannot be null");
            }
            this.policy = protectionPolicy;

            if (targets == null || targets.length == 0) {
                this.targets = null;
            } else {
                this.targets = (Target[]) targets.clone();
            }
        }

        /**
         * Get the array of layer-specific target descriptors that identify
         * the one or more message parts to which the specified message
         * protection policy applies.
         *
         * @return An array of <i>Target</i> that identify targets
         * within a message. This method returns null when the specified policy
         * applies to the whole message (excluding any metadata added to the
         * message to satisfy the policy). This method never returns a
         * zero-length array.
         */
        public Target[] getTargets() {
            return (Target[]) this.targets;
        }

        /**
         * Get the ProtectionPolicy that applies to the targets.
         *
         * @return A ProtectionPolicy object that identifies the
         * message authentication requirements that apply to the
         * targets.
         */
        public ProtectionPolicy getProtectionPolicy() {
            return this.policy;
        }
    }

    /**
     * This interface is used to represent and perform
     * message targeting. Targets are used by message authentication
     * modules to operate on the corresponding content within messages.
     *
     * <p />
     *
     * The internal state of a Target indicates whether it applies
     * to the request or response message of a MessageInfo and to which
     * components it applies within the identified message.
     */
    public static interface Target {

        /**
         * Get the Object identified by the Target from the MessageInfo.
         *
         * @param messageInfo The MessageInfo containing the request or
         *      response message from which the target is to be obtained.
         *
         * @return An Object representing the target,
         *	or null when the target could not be found in the MessageInfo.
         */
        public Object get(MessageInfo messageInfo);

        /**
         * Remove the Object identified by the Target from the MessageInfo.
         *
         * @param messageInfo The MessageInfo containing the request or
         *	response message from which the target is to be removed.
         */
        public void remove(MessageInfo messageInfo);

        /**
         * Put the Object into the MessageInfo at the location identified
         * by the target.
         *
         * @param messageInfo The MessageInfo containing the request or
         *	response message into which the object is to be put.
         */
        public void put(MessageInfo messageInfo, Object data);
    }


    /**
     * This interface is used to represent message authentication policy.
     *
     * <p />
     *
     * The internal state of a ProtectionPolicy object defines the
     * message authentication requirements to be applied to the associated
     * Target.
     */
    public static interface ProtectionPolicy {

        /**
         * The identifier for a ProtectionPolicy that indicates that the
         * sending entity is to be authenticated.
         */
        public static final String AUTHENTICATE_SENDER =
                "#authenticateSender";
        /**
         * The identifier for a ProtectionPolicy that indicates that the
         * origin of data within the message is to be authenticated
         * (that is, the message is to be protected such that
         * its recipients can establish who defined the message
         * content).
         */
        public static final String AUTHENTICATE_CONTENT =
                "#authenticateContent";
        /**
         * The identifier for a ProtectionPolicy that indicates that the
         * message recipient is to be authenticated.
         */
        public static final String AUTHENTICATE_RECIPIENT =
                "#authenticateRecipient";

        /**
         * Get the ProtectionPolicy identifier. An identifier may represent
         * a conceptual protection policy (as is the case with the static
         * identifiers defined within this interface) or it may identify
         * a procedural policy expression or plan that may be more difficult
         * to categorize in terms of a conceptual identifier.
         *
         * @return A String containing a policy identifier. This interface
         *         defines some policy identifier constants.
         *         Configuration systems may define and employ other
         *         policy identifiers values.
         */
        public String getID();
    }
}
