/**
 * EasyBeans
 * Copyright (C) 2010 Bull S.A.S.
 * Contact: easybeans@objectweb.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: CallerPrincipalCallback.java 6201 2012-03-21 10:28:10Z benoitf $
 * --------------------------------------------------------------------------
 */
package javax.security.auth.message.callback;

import java.security.Principal;
import javax.security.auth.Subject;

/**
 * Callback for setting the container's caller (or Remote user) principal.
 * This callback is intended to be called by a <code>serverAuthModule</code>
 * during its <code>validateRequest</code> processing.
 *
 * @version 1.0
 */
public class CallerPrincipalCallback implements javax.security.auth.callback.Callback {
    private static final long serialVersionUID = 7722346085032166879L;

    /**
     * The subject to be MODIFIED BY the callback handler.
     */
    private Subject theSubject;

    /**
     * The principal, used by the callback handler to create the caller
     * principal and add it to the subject.
     * If the principal is stored, then the name isn't.
     */
    private Principal thePrincipal;

    /**
     * The caller name, used by the callback handler to create the caller
     * principal and add it to the subject.
     * If the principal is stored, then the name isn't.
     */
    private String theName;


    /**
     * Create a CallerPrincipalCallback to set the container's
     * representation of the caller principal
     *
     * @param s The Subject in which the container will distinguish the
     * caller identity.
     *
     * @param p The Principal that will be distinguished as the caller
     * principal. This value may be null.
     * <p />
     * The CallbackHandler must use the argument Principal to establish the caller
     * principal associated with the invocation being processed by the container.
     * When the argument Principal is null, the handler must establish the
     * container's representation of the unauthenticated caller principal. The
     * handler may perform principal mapping of non-null argument Principal
     * values, but it must be possible to configure the handler such that it
     * establishes the non-null argument Principal as the caller principal.
     */
    public CallerPrincipalCallback(Subject s, Principal p) {
        this.theSubject = s;
        this.thePrincipal = p;
        this.theName = null;
    }

    /**
     * Create a CallerPrincipalCallback to set the container's
     * representation of the caller principal.
     *
     * @param s The Subject in which the container will distinguish the
     * caller identity.
     *
     * @param n The String value that will be returned when getName() is
     * called on the principal established as the caller principal or null.
     * <p />
     *  The CallbackHandler must use the n argument to establish the caller
     * principal associated with the invocation being processed by the container.
     * When the n argument is null, the handler must establish the container's
     * representation of the unauthenticated caller principal (which may or may
     * not be equal to null, depending on the requirements of the container type
     * ). The handler may perform principal mapping of non-null values of n, but
     * it must be possible to configure the handler such that it establishes the
     * non-null argument value as the value returned when getName is called on
     * the established principal.
     */
    public CallerPrincipalCallback(Subject s, String n) {
        this.theSubject = s;
        this.thePrincipal = null;
        this.theName = n;
    }

    /**
     * Get the Subject in which the handler will distinguish the caller
     * principal
     *
     * @return The subject.
     */
    public Subject getSubject() {
        return theSubject;
    }

    /**
     * Get the caller principal.
     *
     * @return The principal or null.
     * <p />
     * When the values returned by this method and the getName methods
     * are null, the handler must establish the container's representation
     * of the unauthenticated caller principal within the Subject.
     */
    public Principal getPrincipal() {
        return thePrincipal;
    }

    /**
     * Get the caller principal name.
     *
     * @return The principal name or null.
     * <p />
     * When the values returned by this method and the getPrincipal methods
     * are null, the handler must establish the container's representation
     * of the unauthenticated caller principal within the Subject.
     */
    public String getName() {
        return theName;
    }

}
