/**
 * EasyBeans
 * Copyright (C) 2006 Bull S.A.S.
 * Contact: easybeans@objectweb.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: JoinColumn.java 4359 2008-12-10 10:24:40Z sauthieg $
 * --------------------------------------------------------------------------
 */

package javax.persistence;

import static java.lang.annotation.ElementType.FIELD;
import static java.lang.annotation.ElementType.METHOD;
import static java.lang.annotation.RetentionPolicy.RUNTIME;

import java.lang.annotation.Retention;
import java.lang.annotation.Target;

/**
 * The JoinColumn annotation is used to specify a mapped column for joining an
 * entity association.
 * @see <a href="http://www.jcp.org/en/jsr/detail?id=220">EJB 3.0 specification</a>
 * @author Florent Benoit
 * @since EJB 3.0 version.
 */
// TODO: FIXME: Well, TYPE target is not specified in the spec but it is present in the javadoc...
@Target({ METHOD, FIELD })
@Retention(RUNTIME)
public @interface JoinColumn {

    /**
     * (Optional) The name of the foreign key column. The table in which it is
     * found depends upon the context. If the join is for a OneToOne or Many-
     * ToOne mapping, the foreign key column is in the table of the source
     * entity. If the join is for a ManyToMany, the foreign key is in a join
     * table.
     */
    String name() default "";

    /**
     * (Optional) The name of the column referenced by this foreign key column.
     * When used with relationship mappings, the referenced column is in the
     * table of the target entity. When used inside a JoinTable annotation, the
     * referenced key column is in the entity table of the owning entity, or
     * inverse entity if the join is part of the inverse join definition.
     */
    String referencedColumnName() default "";

    /**
     * (Optional) Whether the property is a unique key. This is a shortcut for
     * the UniqueConstraint annotation at the table level and is useful for when
     * the unique key constraint is only a single field. It is not necessary to
     * explicitly specify this for a join column that corresponds to a primary
     * key that is part of a foreign key.
     */
    boolean unique() default false;

    /**
     * (Optional) Whether the foreign key column is nullable.
     */
    boolean nullable() default true;

    /**
     * (Optional) Whether the column is included in SQL INSERT statements
     * generated by the persistence provider.
     */
    boolean insertable() default true;

    /**
     * (Optional) Whether the column is included in SQL UPDATE statements
     * generated by the persistence provider.
     */
    boolean updatable() default true;

    /**
     * (Optional) The SQL fragment that is used when generating the DDL for the
     * column.
     */
    String columnDefinition() default "";

    /**
     * (Optional) The name of the table that contains the column. If a table is
     * not specified, the column is assumed to be in the primary table of the
     * applicable entity.
     */
    String table() default "";

}
