/**
 * OW2 Specifications
 * Copyright (C) 2010 Bull S.A.S.
 * Contact: easybeans@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: PersistenceUnitUtil.java 5417 2010-03-24 09:04:14Z benoitf $
 * --------------------------------------------------------------------------
 */

package javax.persistence;


/**
 * The PersistenceUnitUtil interface provides access to utility methods that can be invoked on entities associated with the
 * persistence unit. The behavior is undefined if these methods are invoked on an entity instance that is not associated with the
 * persistence unit from whose entity manager factory this interface has been obtained.
 * @see <a href="http://jcp.org/en/jsr/detail?id=317">JPA 2.0 specification</a>
 * @author Florent Benoit
 * @since JPA 2.0 version.
 */
public interface PersistenceUnitUtil extends PersistenceUtil {

    /**
     * Determine the load state of a given persistent attribute of an entity belonging to the persistence unit.
     * @param entity entity instance containing the attribute
     * @param attributeName name of attribute whose load state is to be determined
     * @return false if entity's state has not been loaded or if the attribute state has not been loaded, else true
     */
    boolean isLoaded(Object entity, String attributeName);

    /**
     * Determine the load state of an entity belonging to the persistence unit. This method can be used to determine the load
     * state of an entity passed as a reference. An entity is considered loaded if all attributes for which FetchType EAGER has
     * been specified have been loaded. The isLoaded(Object, String) method should be used to determine the load state of an
     * attribute. Not doing so might lead to unintended loading of state.
     * @param entity entity whose load state is to be determined
     * @return false if the entity has not been loaded, else true
     */
    boolean isLoaded(Object entity);

    /**
     * Return the id of the entity. A generated id is not guaranteed to be available until after the database insert has occurred.
     * Returns null if the entity does not yet have an id.
     * @param entity entity instance
     * @return id of the entity
     * @throws IllegalArgumentException if the object is found not to be an entity
     */
    Object getIdentifier(Object entity);
}
