/**
 * OW2 Specifications
 * Copyright (C) 2010 Bull S.A.S.
 * Contact: easybeans@ow2.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
 * USA
 *
 * --------------------------------------------------------------------------
 * $Id: IdentifiableType.java 5417 2010-03-24 09:04:14Z benoitf $
 * --------------------------------------------------------------------------
 */

package javax.persistence.metamodel;

import java.util.Set;

/**
 * Instances of the type IdentifiableType represent entity or mapped superclass types.
 * @param <X> The represented entity or mapped superclass type.
 * @see <a href="http://jcp.org/en/jsr/detail?id=317">JPA 2.0 specification</a>
 * @author Florent Benoit
 * @since JPA 2.0 version.
 */
public interface IdentifiableType<X> extends ManagedType<X> {

    /**
     * Return the attribute that corresponds to the id attribute of the entity or mapped superclass.
     * @param type the type of the represented id attribute
     * @return id attribute
     * @throws IllegalArgumentException if id attribute of the given type is not present in the identifiable type or if the
     * identifiable type has an id class
     */
    <Y> SingularAttribute<? super X, Y> getId(Class<Y> type);

    /**
     * Return the attribute that corresponds to the id attribute declared by the entity or mapped superclass.
     * @param type the type of the represented declared id attribute
     * @return declared id attribute
     * @throws IllegalArgumentException if id attribute of the given type is not declared in the identifiable type or if the
     * identifiable type has an id class
     */
    <Y> SingularAttribute<X, Y> getDeclaredId(Class<Y> type);

    /**
     * Return the attribute that corresponds to the version attribute of the entity or mapped superclass.
     * @param type the type of the represented version attribute
     * @return version attribute
     * @throws IllegalArgumentException if version attribute of the given type is not present in the identifiable type
     */
    <Y> SingularAttribute<? super X, Y> getVersion(Class<Y> type);

    /**
     * Return the attribute that corresponds to the version attribute declared by the entity or mapped superclass.
     * @param type the type of the represented declared version attribute
     * @return declared version attribute
     * @throws IllegalArgumentException if version attribute of the type is not declared in the identifiable type
     */
    <Y> SingularAttribute<X, Y> getDeclaredVersion(Class<Y> type);

    /**
     * Return the identifiable type that corresponds to the most specific mapped superclass or entity extended by the entity or
     * mapped superclass.
     * @return supertype of identifiable type or null if no such supertype
     */
    IdentifiableType<? super X> getSupertype();

    /**
     * Whether the identifiable type has a single id attribute. Returns true for a simple id or embedded id; returns false for an
     * idclass.
     * @return boolean indicating whether the identifiable type has a single id attribute
     */
    boolean hasSingleIdAttribute();

    /**
     * Whether the identifiable type has a version attribute.
     * @return boolean indicating whether the identifiable type has a version attribute
     */
    boolean hasVersionAttribute();

    /**
     * Return the attributes corresponding to the id class of the identifiable type.
     * @return id attributes
     * @throws IllegalArgumentException if the identifiable type does not have an id class
     */
    Set<SingularAttribute<? super X, ?>> getIdClassAttributes();

    /**
     * Return the type that represents the type of the id.
     * @return type of id
     */
    Type<?> getIdType();

}
